;****************************************************************
;*  gbemu.s
;*			Emulation of the Gameboy CPU
;*
;*  2000-02-29  Bodo Wenzel  Creation
;*  2000-04-28  Bodo Wenzel  Several emulations in one executable
;*  2000-05-04  Bodo Wenzel  Squeezing code snippets
;*  2000-06-06  Bodo Wenzel  New memory management
;*  2000-09-09  Bodo Wenzel  Some improvements
;*  2000-09-27  Bodo Wenzel  DAA implemented
;*  2000-10-03  Bodo Wenzel  Interrupts
;*  2001-01-28  Bodo Wenzel  Correction at debug modes
;*  2001-02-05  Bodo Wenzel  New stack management
;****************************************************************
;
;  (c)2000 Bodo Wenzel
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
;****************************************************************

;================================================================
; The emulator is called with everything set up externally.

;================================================================
;	General constants

	include	"constant.s"

;================================================================
; The registers of cpus are mapped that way:
;	Motorola	Gameboy
;	d0		(scratch) data byte
;	d1		(scratch) lower address byte
;	d2		(scratch) higher address byte
;	d3		h(23..16) / l(7..0)
;	d4		d(23..16) / e(7..0)
;	d5		b(23..16) / c(7..0)
;	d6		flags(31..16) / opcode timer (15..0)
;	d7		a(7..0)
;	a0		pc as 68000 address
;	a1		sp as 68000 address
;	a2		(scratch) base pointer for memory access
;	a3		(scratch) pointer for memory access
;	a4		pointer to "gbemu"
;	a5		(not usable, pointer for support)
;	a6		(scratch) temporary return address
;	a7		(not usable, sp for 68000)

;================================================================
;	Special things

	mc68000			;This is ONLY for 68000 cpus!

;================================================================
;	Constants

OFFSET_OP	equ	$00		;the code snippets are
OFFSET_CB	equ	OFFSET_OP+$58	; interleaved this way
OFFSET_RD	equ	OFFSET_CB+$3c	; on pages of 256 bytes
OFFSET_WR	equ	OFFSET_RD+$0e
OFFSET_IN	equ	OFFSET_WR+$10
OFFSET_OUT	equ	OFFSET_IN+$0c
;limit by PalmOS:	$d0

DONT_OPT_HI	equ	$5500	;otherwise the offsets will be
DONT_OPT_LO	equ	$55	;optimized to non-functioning

FLAG_Z		equ	2+16	;masks for Motorola status
FLAG_C		equ	4+16
MASK_Z		equ	1<<(FLAG_Z-16)
MASK_NC		equ	!(1<<(FLAG_C-16))

CPU_Z		equ	-$80	;mask for GB CPU
CPU_C		equ	$10

;================================================================

	text

;================================================================
;	Macros for less typing

;----------------------------------------------------------------
;	administration

macro	return	ret		;return to caller with code
	moveq	#ret,d0
	rts
	endm

macro	exec_pc			;execute next instruction
	local	go_on,jump
	dbra	d6,go_on
	jsr	V_LCD_LINE(a4)
go_on:	move.b	(a0)+,jump+2-gbemu(a4)
jump:	jmp	OFFSET_OP+DONT_OPT_HI(a4)
	endm

macro	next_pc			;jump to next instruction
	if	DEBUG & DEBUG_STEP
	local	go_on
	dbra	d6,go_on
	jsr	V_LCD_LINE(a4)
go_on:	return	SINGLE_STEP
	else
	exec_pc
	endif
	endm

;----------------------------------------------------------------
;	jumps and calls

macro	do_jp			;get jump address
	local	go_on
	move.b	(a0)+,d1
	if	DEBUG & DEBUG_JP32K
	bmi.s	go_on
	move.b	(a0),d0
	addq.b	#1,d0
	bcc.s	go_on
	addq.w	#1,a0
	return	ERR_DEBUG
	endif
go_on:	mem_rd	(a0)+
	move.l	a2,CPU_STATE_PC_B(a4)
	endm

macro	do_push	addr		;push address
	sp_push
	move.l	addr,d0
	sub.l	CPU_STATE_PC_B(a4),d0
	move.w	d0,-(sp)
	move.b	(sp)+,-(a1)
	move.b	d0,-(a1)
	endm

macro	do_call			;push address and get call addr
	do_push	a3
	do_jp
	endm

macro	do_ret			;get return address from stack
	local	go_on
	move.b	(a1)+,d1
	if	DEBUG & DEBUG_JP32K
	bmi.s	go_on
	move.b	(a1),d0
	addq.b	#1,d0
	bcc.s	go_on
	addq.w	#1,a1
	sp_pop
	return	ERR_DEBUG
	endif
go_on:	mem_rd	(a1)+
	move.l	a2,CPU_STATE_PC_B(a4)
	move.l	a3,a0
	sp_pop
	endm

;----------------------------------------------------------------
;	memory and input/output

macro	mem_rd	hi		;read memory (or input, hi=$ff)
	local	jump,ret
	lea	ret-gbemu(a4),a6
	move.b	hi,jump+2-gbemu(a4)
jump:	jmp	OFFSET_RD+DONT_OPT_HI(a4)
ret:
	endm

macro	mem_wr	hi		;write memory (or output, hi=$ff)
	local	jump,ret
	lea	ret-gbemu(a4),a6
	move.b	hi,jump+2-gbemu(a4)
jump:	jmp	OFFSET_WR+DONT_OPT_HI(a4)
ret:
	endm

macro	io_rd	lo		;read input
	local	jump,ret
	lea	ret-gbemu(a4),a6
	move.b	lo,jump+2-gbemu(a4)
jump:	jmp	OFFSET_IN+DONT_OPT_HI(a4)
ret:
	endm

macro	io_wr	lo		;write output
	local	jump,ret
	lea	ret-gbemu(a4),a6
	move.b	lo,jump+2-gbemu(a4)
jump:	jmp	OFFSET_OUT+DONT_OPT_HI(a4)
ret:
	endm

macro	sp_push			;check and correct pushing sp
	local	go_on
	subq.b	#2,CPU_STATE_SP+1(a4)
	bcc.s	go_on
	jsr	SP_PUSH_CORRECT(a4)
	;Here could be a check for writing into rom...
go_on:
	endm

macro	sp_pop			;check and correct popping sp
	local	go_on
	addq.b	#2,CPU_STATE_SP+1(a4)
	bcc.s	go_on
	jsr	SP_POP_CORRECT(a4)
go_on:
	endm

;================================================================
;	Macros for opcodes

	include	"gbemu_op.s"

;================================================================
;	Macros for opcodes after prefix byte $cb

	include	"gbemu_cb.s"

;================================================================
;	Macros for reading and writing memory

	include	"gbemu_rw.s"

;================================================================
;	Macros for input and output

	include	"gbemu_io.s"

;================================================================
;	Macros for extensions

	include	"gbemu_xt.s"

;================================================================
;	And now produce the code!

macro	combo	num
	if	DEBUG & DEBUG_PROFILE
	local	prof
	endif
	org	((num ^ $80) << 8) + OFFSET_OP
	if	DEBUG & DEBUG_PROFILE
	addq.l	#1,prof-gbemu(a4)
	endif
	op&num
	org	((num ^ $80) << 8) + OFFSET_CB
	cb&num
	org	((num ^ $80) << 8) + OFFSET_RD
	rd&num
	org	((num ^ $80) << 8) + OFFSET_WR
	wr&num
	org	((num ^ $80) << 8) + OFFSET_IN
	in&num
	org	((num ^ $80) << 8) + OFFSET_OUT
	out&num
	if	num == $7f
	end_label
	endif
	org	((num ^ $80) << 8) + OFFSET_XT
	xt&num
	if	DEBUG & DEBUG_PROFILE
	org	((num ^ $80) << 8) + OFFSET_PROF
prof:	dc.l	0
	endif
	endm

;----------------------------------------------------------------

	org	$0000		;first codes $80 to $ff
	begin_label

i	set	$80

	rept	$80
	combo	$(i)
i	set	i+1
	endm

	org	$8000		;now codes $00 to $7f

gbemu:
	base_label		;MAIN LABEL (this is a macro)
				;"The Middle of the Code"
				;that way we can reach full 64K

i	set	$00

	rept	$80
	combo	$(i)
i	set	i+1
	endm

;================================================================
	end
