;****************************************************************
;*  gbemu_io.s
;*			Emulation of the Gameboy CPU
;*			(Input and output)
;*
;*  2000-06-18  Bodo Wenzel  Creation
;*  2000-09-09  Bodo Wenzel  Now realy full screen emulation
;*  2000-09-29  Bodo Wenzel  Some improvements in source text
;*  2000-10-15  Bodo Wenzel  Interrupts and timer
;*  2000-11-16  Bodo Wenzel  Better joypad emulation
;*  2001-01-28  Bodo Wenzel  Dummy SC emulation
;****************************************************************
;
;  (c)2000 Bodo Wenzel
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
;****************************************************************

;================================================================
; All i/o accesses are generated with:
;	d0.b	byte to be written
;	d1.b	(io_rd: must be preserved)
;	d2.b	(io_rd: must be preserved)
;	a2	68000 base memory address (used after io_rd)
;	a2	68000 base memory address
;	a3	68000 memory address (used after io_rd,
;		  also as "io_lea")
;	a6	68000 return address

;================================================================
; Macros for generating macros...

macro	in_ign	num
macro	in&num			;not emulated and ignored
	lea	io&num-gbemu(a4),a3
	jmp	(a6)
	endm
	endm

macro	out_ign	num
macro	out&num			;not emulated and ignored
	move.b	d0,io&num-gbemu(a4)	;but byte saved
	jmp	(a6)
io&num:	dc.b	$ff
	endm
	endm

macro	out_nul	num
macro	out&num			;not emulated and ignored
	jmp	(a6)
io&num:	dc.b	$ff
	endm
	endm

macro	in_err	num
	if	DEBUG & DEBUG_NO_RW
macro	in&num			;not emulated and error
	return	NO_INP
	endm
	else
	in_ign	num
	endif
	endm

macro	out_err	num
	if	DEBUG & DEBUG_NO_RW
macro	out&num			;not emulated and error
	return	NO_OUT
	endm
	else
	out_nul	num
	endif
	endm

macro	in_ram	num		;CPU RAM read
macro	in&num
	move.l	BASE_RAM_CPU(a4),a2
	lea	$ff00+num(a2),a3
	jmp	(a6)
	endm
	endm

macro	out_ram	num		;CPU RAM write
macro	out&num
	move.l	BASE_RAM_CPU(a4),a2
	move.b	d0,$ff00+num(a2)
	jmp	(a6)
	endm
	endm

;================================================================

macro	in$0			;P1 joypad
	move.l	JOYPAD_PTR(a4),a3
	jmp	(a6)
	endm

macro	out$0			;P1 joypad
	lsr.b	#4,d0
	and.w	#3,d0
	lea	JOYPAD_BOTH(a4),a3
	add.w	d0,a3
	move.l	a3,JOYPAD_PTR(a4)
	jmp	(a6)
	endm

macro	in$1			;SB serial transfer data
	return	NO_INP
	endm
macro	in$1		;***	;SB serial transfer data
	lea	io$1-gbemu(a4),a3
	jmp	(a6)
io$1:	dc.b	$ff
	endm

macro	out$1			;SB serial transfer data
	return	NO_OUT
	endm
macro	out$1		;***	;SB serial transfer data
	jmp	(a6)
io$2m:	bclr	#7,d0
	bset	#3,INT_IF(a4)
	bra	io$2q
	endm

macro	in$2			;SC serial transfer control
	return	NO_INP
	endm
macro	in$2		;***	;SC serial transfer control
	lea	io$2-gbemu(a4),a3
	jmp	(a6)
io$2:	dc.b	$00
	endm

macro	out$2			;SC serial transfer control
	return	NO_OUT
	endm
macro	out$2		;***	;SC serial transfer control
	btst	#0,d0
	bne	io$2m
io$2q:	;macht assembler falsch, wenn der move hier steht...
	move.b	d0,io$2-gbemu(a4)
	jmp	(a6)
	endm

	in_err	$3		;$03 (not existing)
	out_err	$3

macro	in$4			;DIV divider
	lea	T_DIV(a4),a3
	jmp	(a6)
	endm

macro	out$4			;DIV divider
	move.w	#0,T_DIV(a4)
	jmp	(a6)
	endm

macro	in$5			;TIMA timer counter
	lea	T_TIMA(a4),a3
	jmp	(a6)
	endm

macro	out$5			;TIMA timer counter
	lea	T_TIMA(a4),a3
	move.w	#0,(a3)
	move.b	d0,(a3)
	jmp	(a6)
	endm

macro	in$6			;TMA timer modulo
	lea	T_TMA(a4),a3
	jmp	(a6)
	endm

macro	out$6			;TMA timer modulo
	move.b	d0,T_TMA(a4)
	jmp	(a6)
	endm

macro	in$7			;TAC timer control
	lea	io$7-gbemu(a4),a3
	jmp	(a6)
io$7:	dc.b	0
	endm

macro	out$7			;TAC timer control
	move.b	d0,io$7-gbemu(a4)
	jmp	IO_WR_TAC(a4)
	endm

i	set	$8
	rept	$f-i
	in_err	$(i)		;$08..$0e (not existing)
	out_err	$(i)
i	set	i+1
	endm

macro	in$f			;IF interrupt flag
	lea	INT_IF(a4),a3
	jmp	(a6)
	endm

macro	out$f			;IF interrupt flag
	move.b	d0,INT_IF(a4)
	jmp	(a6)
	endm

	in_ign	$10		;CH1_ENT (sound)
	out_ign	$10		;CH1_ENT (sound)

	in_ign	$11		;CH1_WAVE (sound)
	out_ign	$11		;CH1_WAVE (sound)

	in_ign	$12		;CH1_ENV (sound)
	out_ign	$12		;CH1_ENV (sound)

	in_err	$13		;$13 (not existing)
	out_nul	$13		;CH1_FREQ_LO (sound)

	in_ign	$14		;CH1_FREQ_HI_KICK (sound)
	out_ign	$14		;CH1_FREQ_HI_KICK (sound)

	in_err	$15		;$15 (not existing)
	out_err	$15

	in_ign	$16		;CH2_WAVE (sound)
	out_ign	$16		;CH2_WAVE (sound)

	in_ign	$17		;CH2_ENV (sound)
	out_ign	$17		;CH2_ENV (sound)

	in_err	$18		;$18 (not existing)
	out_nul	$18		;CH2_FREQ_LO (sound)

	in_ign	$19		;CH2_FREQ_HI_KICK (sound)
	out_ign	$19		;CH2_FREQ_HI_KICK (sound)

	in_ign	$1a		;CH3_ONOFF (sound)
	out_ign	$1a		;CH3_ONOFF (sound)

	in_ign	$1b		;CH3_LENGTH (sound)
	out_ign	$1b		;CH3_LENGTH (sound)

	in_ign	$1c		;CH3_VOLUME (sound)
	out_ign	$1c		;CH3_VOLUME (sound)

	in_err	$1d		;$1d (not existing)
	out_nul	$1d		;CH3_FREQ_LO (sound)

	in_ign	$1e		;CH3_FREQ_HI_KICK (sound)
	out_ign	$1e		;CH3_FREQ_HI_KICK (sound)

	in_err	$1f		;$1f (not existing)
	out_err	$1f

	in_ign	$20		;CH4_LENGTH (sound)
	out_ign	$20		;CH4_LENGTH (sound)

	in_ign	$21		;CH4_ENV (sound)
	out_ign	$21		;CH4_ENV (sound)

	in_ign	$22		;CH4_POLY (sound)
	out_ign	$22		;CH4_POLY (sound)

	in_ign	$23		;CH4_KICK (sound)
	out_ign	$23		;CH4_KICK (sound)

	in_ign	$24		;SND_VIN (sound)
	out_ign	$24		;SND_VIN (sound)

	in_ign	$25		;SND_STEREO (sound)
	out_ign	$25		;SND_STEREO (sound)

	in_ign	$26		;SND_STAT (sound)
	out_ign	$26		;SND_STAT (sound)

i	set	$27
	rept	$30-i
	in_err	$(i)		;$27..$2f (not existing)
	out_err	$(i)
i	set	i+1
	endm

i	set	$30		;RAM wave pattern
	rept	$40-i
	in_ign	$(i)
	out_ign	$(i)
i	set	i+1
	endm

macro	in$40			;LCDC LCD control
	lea	LCD_LCDC(a4),a3
	jmp	(a6)
	endm

macro	out$40			;LCDC LCD control
	lea	LCD_LCDC(a4),a3
	jmp	V_IO_WR_LCDC(a4)
	endm

macro	in$41			;STAT LCD status
	lea	LCD_STAT(a4),a3
	jmp	V_IO_RD_STAT(a4)
	endm

macro	out$41			;STAT LCD status
	lea	LCD_STAT(a4),a3
	jmp	V_IO_WR_STAT(a4)
	endm

macro	in$42			;SCY scroll Y
	lea	LCD_SCY(a4),a3
	jmp	(a6)
	endm

macro	out$42			;SCY scroll Y
	lea	LCD_SCY(a4),a3
	jmp	V_IO_WR_SCY(a4)
	endm

macro	in$43			;SCX scroll X
	lea	LCD_SCX(a4),a3
	jmp	(a6)
	endm

macro	out$43			;SCX scroll X
	lea	LCD_SCX(a4),a3
	jmp	V_IO_WR_SCX(a4)
	endm

macro	in$44			;LY LCD Y coordinate
	lea	LCD_LY-1(a4),a3
	moveq	#143-1-$100,d0
	sub.b	(a3)+,d0
	move.b	d0,(a3)
	jmp	(a6)
	endm

	out_err	$44		;LY LCD Y coordinate
	;it's not clear if it can be reset by writing...

macro	in$45			;LYC LCD Y compare
	lea	LCD_LYC(a4),a3
	jmp	(a6)
	endm

macro	out$45			;LYC LCD Y compare
	lea	LCD_LYC(a4),a3
	jmp	V_IO_WR_LYC(a4)
	endm

	in_err	$46

macro	out$46			;DMA transfer and start address
	local	jump
	pea	(a6)
	clr.b	d1
	move.l	V_IO_WR_DMA+2(a4),a6
	move.b	d0,jump+2-gbemu(a4)
jump:	jmp	OFFSET_RD+DONT_OPT_HI(a4)
io$46:	dc.b	$ff
	endm

macro	in$47			;BGP background palette
	lea	LCD_BGP(a4),a3
	jmp	(a6)
	endm

macro	out$47			;BGP background palette
	lea	LCD_BGP(a4),a3
	jmp	V_IO_WR_BGP(a4)
	endm

macro	in$48			;OBP0 object palette 0
	lea	LCD_OBP0(a4),a3
	jmp	(a6)
	endm

macro	out$48			;OBP0 object palette 0
	lea	LCD_OBP0(a4),a3
	jmp	V_IO_WR_OBP0(a4)
	endm

macro	in$49			;OBP1 object palette 1
	lea	LCD_OBP1(a4),a3
	jmp	(a6)
	endm

macro	out$49			;OBP1 object palette 1
	lea	LCD_OBP1(a4),a3
	jmp	V_IO_WR_OBP1(a4)
	endm

macro	in$4a			;WY window Y
	lea	LCD_WY(a4),a3
	jmp	(a6)
	endm

macro	out$4a			;WY window Y
	lea	LCD_WY(a4),a3
	jmp	V_IO_WR_WY(a4)
	endm

macro	in$4b			;WX window X
	lea	LCD_WX(a4),a3
	jmp	(a6)
	endm

macro	out$4b			;WX window X
	lea	LCD_WX(a4),a3
	jmp	V_IO_WR_WX(a4)
	endm

i	set	$4c
	rept	$80-i
	in_err	$(i)		;$4c..$7f (not existing)
	out_err	$(i)
i	set	i+1
	endm

i	set	$80
	rept	$7f
	in_ram	$(i)		;RAM in CPU
	out_ram	$(i)
i	set	i+1
	endm

	in_ram	$ff		;IE interrupt enable
				;has to be in RAM for sp handling
macro	out$ff			;IE interrupt enable
	jmp	IO_WR_IE(a4)
	endm

;================================================================
;	end of include
