/****************************************************************
 *  stemboy.c
 *			Atari ST Gameboy Emulator (TOS)
 *
 *  2000-02-29  Bodo Wenzel  Creation
 *  2000-04-28  Bodo Wenzel  Changed from simple to GEM version
 *  2000-08-29  Bodo Wenzel  Commandline evaluation and much more
 *                           support for breakpoints
 *  2000-09-15  Bodo Wenzel  Now nearly full screen emulation
 *  2000-09-29  Bodo Wenzel  PalmBoy scroller as GB code
 *  2000-10-18  Bodo Wenzel  Autostart
 *  2001-01-17  Bodo Wenzel  Support for color screens (SLOW!)
 *  2001-01-21  Bodo Wenzel  Joystick emulation via keys
 *  2001-02-10  Bodo Wenzel  Debug enhancements
 *  2001-06-20  Matthias Jaap Support for long file names
 ****************************************************************

  (c)2000 Bodo Wenzel

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 ****************************************************************/

/* === Includes ===============================================	*/

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <aes.h>
#include <tos.h>
#include <vdi.h>

#include "stemboy.h"
#include "win_scr.h"
#include "win_cpu.h"
#include "win_prof.h"
#include "cartridg.h"
#include "palm.h"
#include "emulate.h"
#include "gem_add.h"
#include "emu_wrap.h"
#include "version.h"
#define GLOBAL    0x0020
#define ALLOCMODE 3|GLOBAL
#define GS_REQUEST  0x1350
#define GS_REPLY    0x1351
#define GS_COMMAND  0x1352
#define GS_ACK      0x1353
#define GS_QUIT     0x1354

#define GSM_COMMAND 0x0001

#define GSACK_OK      0
#define GSACK_UNKNOWN 1
#define GSACK_ERROR   2
typedef struct
{
	long len;
	int  version;
	int  msgs;
	long ext;
} GS_INFO;
int gsapp=-1;
char *xaccname =NULL;
GS_INFO    *gsi = NULL;
/* === Constants ==============================================	*/

#define  FALSE  (0)
#define  TRUE   (!FALSE)
#define VA_START			0x4711

#define  RSC_FILENAME  "stemboy.rsc"

#define  ALERT_NORSC  "[3][" RSC_FILENAME " ?][Stop]"

/* === M A I N ================================================	*/
int doGSCommand(int pipe[8])
{
	int   answ[8], msg[8],ret=0;
	char *cmd = *(char **)&pipe[3];
	char *gslongname =NULL;
	answ[0]=GS_ACK;
	answ[1]=ap_id;
	answ[2]=0;
	answ[3]=pipe[3];
	answ[4]=pipe[4];
	answ[5]=0;
	answ[6]=0;
	answ[7]=GSACK_ERROR;
	if (xaccname==NULL) {
		xaccname = (char *)Mxalloc(96L,ALLOCMODE);
	}
	gslongname = xaccname+64;
	strcpy(gslongname,"STemBoy");
	if (cmd)
	{
		answ[7]=GSACK_UNKNOWN;

		if ( (!stricmp(cmd,"Quit")) || (!stricmp(cmd,"Shutdown")) )
		{
			ret=1;
			answ[7]=GSACK_OK;
			appl_write(pipe[1],16,answ);
	        msg[3]=MFILE;
         	msg[4]=MFQUIT;
         	menu_tnormal(menu,msg[3],0);
         	appl_write(ap_id,16,msg);
		}
		else if (!stricmp(cmd,"Open")) {
			answ[7]=GSACK_OK;
			appl_write(pipe[1],16,answ);
	        msg[3]=MFILE;
       		msg[4]=MFLOAD;
         	menu_tnormal(menu,msg[3],0);
         	appl_write(ap_id,16,msg);
		} else if (!stricmp(cmd,"AppGetLongName")) {
			if (gslongname)
			{
				answ[5]=(int)(((long)gslongname >> 16) & 0x0000ffffL);
				answ[6]=(int)((long)gslongname & 0x0000ffffL);
				answ[7]=GSACK_OK;
			}
			else
			{
				answ[7]=GSACK_ERROR;
			}
			appl_write(pipe[1],16,answ);
		}
	}

	return(ret);
}
int main(int argc,char *argv[]) {
  /*---------------------------*/
  int        work_in[11]={ 1,1,1,1,1,1,1,0,1,1,2 };
  int        work_out[57];
  int        event,msg[8],mx,my,kstate,kchar;
  int        whoosh,ping,doit,dummy,top;
  OBSPEC     *alert;
  int        me_which;
  EMU_COMBO  *emu_which;
  int        scr_handle,cpu_handle,prf_handle;
  Pdomain(1);
  if ((ap_id=appl_init())<0)
    return 1;

  graf_mouse(BUSYBEE,NULL);

  vdi_handle=graf_handle(&wchr,&hchr,&dummy,&dummy);
  v_opnvwk(work_in,&vdi_handle,work_out);
  if (vdi_handle==0) {
    graf_mouse(ARROW,NULL);
    appl_exit();
    return 1;
  }

  if (!rsrc_load(RSC_FILENAME)) {
    graf_mouse(ARROW,NULL);
    form_alert(1,ALERT_NORSC);
    v_clsvwk(vdi_handle);
    appl_exit();
    return 1;
  }

  vst_alignment(vdi_handle,0,5,&dummy,&dummy);
  vsf_perimeter(vdi_handle,0);

  graf_mouse(ARROW,NULL);
  wind_update(BEG_UPDATE);
  graf_mouse(M_OFF,NULL);

  rsrc_gaddr(R_TREE,MENU,&menu);
  menu_bar(menu,1);

  emu_init();
  me_which=MECHECK;
  emu_which=emu_chk;
  emu_reset(emu_which);

  if ((scr_handle=scr_open())<0) {
    wind_update(END_UPDATE);
    graf_mouse(M_ON,NULL);
    rsrc_gaddr(R_FRSTR,ASCREEN,&alert);
    form_alert(1,alert->free_string);
    v_clsvwk(vdi_handle);
    appl_exit();
    return 1;
  }

  cpu_handle=-1;
  cpu_free_lines(cpu_handle);

  prf_handle=-1;
  prf_update(prf_handle,emu_which,TRUE);

  graf_mouse(M_ON,NULL);

  whoosh=1;
  ping=0;

  msg[0]=MN_SELECTED;
  msg[1]=ap_id;
  msg[2]=0;
  msg[3]=0;
  msg[4]=0;
  appl_write(ap_id,16,msg);

  if (argc>1) {
    menu_tnormal(menu,MFILE,0);
    if (cartr_load(argv[1])) {
      menu_tnormal(menu,MFILE,1);
      menu_tnormal(menu,MEMU,0);
      msg[0]=MN_SELECTED;
      msg[1]=ap_id;
      msg[2]=0;
      msg[3]=MEMU;
      msg[4]=MERUN;
      appl_write(ap_id,16,msg);
    } else
      menu_tnormal(menu,MFILE,1);
  }

/* - - begin of main loop - - - - - - - - - - - - - - - - - - -	*/

  doit=TRUE;
  while (doit) {
    wind_update(END_UPDATE);
    if (scr_double>=0)
      event=evnt_multi(MU_KEYBD|MU_MESAG|MU_TIMER,
                       0,0,0,
                       0,0,0,0,0,
                       0,0,0,0,0,
                       msg,
                       0,0,
                       &mx,&my,&dummy,
                       &kstate,&kchar,&dummy);
    else
      event=evnt_multi(MU_KEYBD|MU_MESAG,
                       0,0,0,
                       0,0,0,0,0,
                       0,0,0,0,0,
                       msg,
                       0,0,
                       &mx,&my,&dummy,
                       &kstate,&kchar,&dummy);
    wind_update(BEG_UPDATE);

    scr_showgray(scr_handle,mx,my);

    if (event & MU_MESAG) {
      switch (msg[0]) {
  
/* - - menu commands  - - - - - - - - - - - - - - - - - - - - -	*/
      case MN_SELECTED:
        switch (msg[4]) {
        case MABOUT:
          program_info();
          break;
          /*form_alert(1,version);
          rsrc_gaddr(R_FRSTR,AKEYMAP,&alert);
          form_alert(1,alert->free_string);
          break; */

        case MFAUTO:
          whoosh=1-whoosh;
          break;

        case MFLOAD:
          if (cartr_load(NULL)) {
            emu_reset(emu_which);
            prf_update(prf_handle,emu_which,TRUE);
            if (whoosh) {
              int  m[8];
              m[0]=MN_SELECTED;
              m[1]=ap_id;
              m[2]=0;
              m[3]=MEMU;
              m[4]=MERUN;
              appl_write(ap_id,16,m);
              menu_tnormal(menu,m[3],0);
            }
          }
          break;

        case MFINFO:
          if (emu_rom==NULL)
            break;
          cartr_info();
          break;

        case MFEXPORT:
          palm_export();
          break;

        case MFQUIT:
          doit=FALSE;
          break;

        case MEPURE:
        case MECHECK:
        case MEPROF:
        case MESSTEP:
          me_which=msg[4];
          switch (me_which) {
          case MECHECK:  emu_which=emu_chk;   break;
          case MEPROF:   emu_which=emu_prof;  break;
          case MESSTEP:  emu_which=emu_step;  break;
          default:       emu_which=emu_pure;  break;
          }
          emu_reset(emu_which);
          prf_update(prf_handle,emu_which,TRUE);
          break;

        case MEANIM:
          if (me_which!=MESSTEP)
            break;
          /* fall through */
        case MERUN:
          if (emu_rom==NULL)
            break;
          graf_mouse(M_OFF,NULL);
          emu_mykbd();
          scr_auto_on(scr_handle);
          for (;;) {
            emu_run(emu_which);
            if (emu_quit!=0 &&
                emu_state.run_result==SINGLE_STEP)
              emu_state.run_result=USER_STOP;
            if (!cpu_add_line(cpu_handle,emu_which,&emu_state,
                   &(emu_which[CPU_STATE].ext.cpu_regs))) {
              doit=FALSE;
              break;
            }
            prf_update(prf_handle,emu_which,FALSE);
            if (msg[4]!=MEANIM)
              break;
            if (emu_state.run_result!=SINGLE_STEP)
              break;
          }
          scr_auto_off(scr_handle);
          emu_oldkbd();
          graf_mouse(M_ON,NULL);
          if (ping!=0)
            Cconout('\a');
          if (!doit) {
            rsrc_gaddr(R_FRSTR,ANOMEM,&alert);
            form_alert(1,alert->free_string);
          }
          break;

        case MEPING:
          ping=1-ping;
          break;

        case MWCPU:
          if (cpu_handle>=0) {
            cpu_close(cpu_handle);
            cpu_handle=-1;
          } else if ((cpu_handle=cpu_open())<0) {
            rsrc_gaddr(R_FRSTR,ANOWIN,&alert);
            form_alert(1,alert->free_string);
          }
          break;

        case MWPROF:
          if (prf_handle>=0) {
            prf_close(prf_handle);
            prf_handle=-1;
          } else if ((prf_handle=prf_open(emu_which))<0) {
            rsrc_gaddr(R_FRSTR,ANOWIN,&alert);
            form_alert(1,alert->free_string);
          }
          break;

        case MWCYCLE:
          wind_get(0,WF_TOP,&top);
          if (top==scr_handle) {
            if (cpu_handle>=0)
              wind_set(cpu_handle,WF_TOP);
            else if (prf_handle>=0)
              wind_set(prf_handle,WF_TOP);
          } else if (top==cpu_handle) {
            if (prf_handle>=0)
              wind_set(prf_handle,WF_TOP);
            else
              wind_set(scr_handle,WF_TOP);
          } else if (top==prf_handle)
            wind_set(scr_handle,WF_TOP);
          break;

        case MWCLOSE:
          wind_get(0,WF_TOP,&top);
          if (top==cpu_handle) {
            cpu_close(cpu_handle);
            cpu_handle=-1;
          } else if (top==prf_handle) {
            prf_close(prf_handle);
            prf_handle=-1;
          }
          break;

        case MDCCLR:
          cpu_free_lines(cpu_handle);
          break;

        case MDPCLR:
          prf_update(prf_handle,emu_which,TRUE);
          break;

        case MDSCROLL:
          emu_try_scroller(emu_which);
          break;

        case MDTRY:
          emu_try_code(emu_which,'\0');
          break;

        case MDSETBP:
          emu_set_bp();
          break;

        case MDDOUBLE:
          if (scr_double>=0)
            scr_double=1-scr_double;
          break;

        default:
          break;
        }

        menu_tnormal(menu,msg[3],1);
        menu_icheck(menu,MFAUTO,whoosh);
        menu_ienable(menu,MFINFO,(emu_rom!=NULL) ? 1 : 0);
        menu_icheck(menu,MEPURE,0);
        menu_icheck(menu,MECHECK,0);
        menu_icheck(menu,MEPROF,0);
        menu_icheck(menu,MESSTEP,0);
        menu_icheck(menu,me_which,1);
        menu_ienable(menu,MERUN,(emu_rom!=NULL) ? 1 : 0);
        if (me_which==MESSTEP && emu_rom!=NULL)
          menu_ienable(menu,MEANIM,1);
        else
          menu_ienable(menu,MEANIM,0);
        menu_icheck(menu,MEPING,ping);
        menu_icheck(menu,MWCPU,(cpu_handle>=0) ? 1 : 0);
        menu_icheck(menu,MWPROF,(prf_handle>=0) ? 1 : 0);
        if (cpu_handle>=0 || prf_handle>=0) {
          menu_ienable(menu,MWCYCLE,1);
          menu_ienable(menu,MWCLOSE,1);
        } else {
          menu_ienable(menu,MWCYCLE,0);
          menu_ienable(menu,MWCLOSE,0);
        }
        menu_ienable(menu,MDSETBP,(emu_rom!=NULL) ? 1 : 0);
        if (scr_double>=0) {
          menu_icheck(menu,MDDOUBLE,scr_double);
          menu_ienable(menu,MDDOUBLE,1);
        } else {
          menu_icheck(menu,MDDOUBLE,1);
          menu_ienable(menu,MDDOUBLE,0);
        }
        break;

/* - - window handling  - - - - - - - - - - - - - - - - - - - -	*/
      case WM_REDRAW:
        if (msg[3]==cpu_handle)
          cpu_redraw(cpu_handle,msg[4],msg[5],msg[6],msg[7]);
        else if (msg[3]==prf_handle)
          prf_redraw(prf_handle,msg[4],msg[5],msg[6],msg[7]);
        break;

      case WM_TOPPED:
      case WM_NEWTOP:
        wind_set(msg[3],WF_TOP);
        break;

      case WM_CLOSED:
        msg[0]=MN_SELECTED;
        msg[1]=ap_id;
        if (msg[3]==cpu_handle)
          msg[4]=MWCPU;
        else if (msg[3]==prf_handle)
          msg[4]=MWPROF;
        appl_write(ap_id,16,msg);
        break;
		case GS_REQUEST:
			{
				int answ[8];
				
				answ[0]=GS_REPLY;
				answ[1]=ap_id;
				answ[2]=0;
				answ[3]=0;
				answ[4]=0;
				answ[5]=0;
				answ[6]=1;
				answ[7]=msg[7];

				if (!gsi) gsi = (GS_INFO *)Mxalloc(sizeof(GS_INFO),ALLOCMODE);
				
				if (gsi)
				{
					GS_INFO *sender = *(GS_INFO **)&msg[3];

					gsi->len     = sizeof(GS_INFO);
					gsi->version = 0x0100;
					gsi->msgs    = GSM_COMMAND;
					gsi->ext     = 0L;
					
					answ[3]=(int)(((long)gsi >> 16) & 0x0000ffffL);
					answ[4]=(int)((long)gsi & 0x0000ffffL);
					
					if (sender)
					{
						if (sender->version >= 0x0070)
						{
							answ[6]=0;
							gsapp=msg[1];
						}
					}
				}
				
				appl_write(gsapp,16,answ);
			}
			break;
		case GS_COMMAND:
			(void) doGSCommand(msg);
			break;

      case WM_FULLED:
        if (msg[3]==cpu_handle)
          cpu_fulled(cpu_handle);
        else if (msg[3]==prf_handle)
          prf_fulled(prf_handle);
        break;

      case WM_ARROWED:
        if (msg[3]==cpu_handle)
          cpu_arrowed(cpu_handle,msg[4]);
        else if (msg[3]==prf_handle)
          prf_arrowed(prf_handle,msg[4]);
        break;

      case WM_VSLID:
        if (msg[3]==cpu_handle)
          cpu_vslided(cpu_handle,msg[4]);
        else if (msg[3]==prf_handle)
          prf_vslided(prf_handle,msg[4]);
        break;

      case WM_HSLID:
        if (msg[3]==cpu_handle)
          cpu_hslided(cpu_handle,msg[4]);
        else if (msg[3]==prf_handle)
          prf_hslided(prf_handle,msg[4]);
        break;

      case WM_SIZED:
        if (msg[3]==cpu_handle)
          cpu_sized(cpu_handle,msg[4],msg[5],msg[6],msg[7]);
        else if (msg[3]==prf_handle)
          prf_sized(prf_handle,msg[4],msg[5],msg[6],msg[7]);
        break;

      case WM_MOVED:
        if (msg[3]==scr_handle)
          scr_moved(scr_handle,msg[4],msg[5]);
        else if (msg[3]==cpu_handle)
          cpu_moved(cpu_handle,msg[4],msg[5],msg[6],msg[7]);
        else if (msg[3]==prf_handle)
          prf_moved(prf_handle,msg[4],msg[5],msg[6],msg[7]);
        break;

      default:
        break;
      }
    }

/* - - key commands - - - - - - - - - - - - - - - - - - - - - -	*/
    if (event & MU_KEYBD) {
      msg[1]=ap_id;
      msg[2]=0;

      switch (kchar>>8) {
      case KEY_HOME:
        msg[0]=WM_VSLID;
        if (kstate&(K_RSHIFT|K_LSHIFT))
          msg[4]=1000;
        else
          msg[4]=0;
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_LEFT:
        if (kstate&(K_RSHIFT|K_LSHIFT)) {
          msg[0]=WM_HSLID;
          msg[4]=0;
        } else {
          msg[0]=WM_ARROWED;
          msg[4]=WA_LFLINE;
        }
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_RIGHT:
        if (kstate&(K_RSHIFT|K_LSHIFT)) {
          msg[0]=WM_HSLID;
          msg[4]=1000;
        } else {
          msg[0]=WM_ARROWED;
          msg[4]=WA_RTLINE;
        }
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_CLEFT:
        msg[0]=WM_ARROWED;
        msg[4]=WA_LFPAGE;
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_CRIGHT:
        msg[0]=WM_ARROWED;
        msg[4]=WA_RTPAGE;
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_UP:
        msg[0]=WM_ARROWED;
        if (kstate&(K_RSHIFT|K_LSHIFT|K_CTRL))
          msg[4]=WA_UPPAGE;
        else
          msg[4]=WA_UPLINE;
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_DOWN:
        msg[0]=WM_ARROWED;
        if (kstate&(K_RSHIFT|K_LSHIFT|K_CTRL))
          msg[4]=WA_DNPAGE;
        else
          msg[4]=WA_DNLINE;
        wind_get(0,WF_TOP,&msg[3]);
        appl_write(ap_id,16,msg);
        break;

      case KEY_R:
        if (kstate&K_ALT) {
          msg[0]=MN_SELECTED;
          msg[3]=MEMU;
          msg[4]=MERUN;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
        }
        break;

      case KEY_A:
        if (kstate&K_ALT) {
          msg[0]=MN_SELECTED;
          msg[3]=MEMU;
          msg[4]=MEANIM;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
        }
        break;

      case KEY_M:
        if (kstate&K_ALT) {
          msg[0]=MN_SELECTED;
          msg[3]=MWINDOW;
          msg[4]=MWCPU;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
        }
        break;

      case KEY_P:
        if (kstate&K_ALT) {
          msg[0]=MN_SELECTED;
          msg[3]=MWINDOW;
          msg[4]=MWPROF;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
        }
        break;

      default:
        msg[0]=MN_SELECTED;
        switch (kchar&0xff) {
        case 'O'&0x1f:
          msg[3]=MFILE;
          msg[4]=MFLOAD;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
          break;

        case 'I'&0x1f:
          msg[3]=MFILE;
          msg[4]=MFINFO;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
          break;

        case 'Q'&0x1f:
          msg[3]=MFILE;
          msg[4]=MFQUIT;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
          break;

        case 'W'&0x1f:
          msg[3]=MWINDOW;
          msg[4]=MWCYCLE;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
          break;

        case 'U'&0x1f:
          msg[3]=MWINDOW;
          msg[4]=MWCLOSE;
          menu_tnormal(menu,msg[3],0);
          appl_write(ap_id,16,msg);
          break;

        default:
          if (isxdigit(kchar&0xff)) {
            menu_tnormal(menu,MDEBUG,0);
            emu_try_code(emu_which,kchar);
            menu_tnormal(menu,MDEBUG,1);
          }
          break;
        }
        break;
      }

      while(evnt_multi(MU_KEYBD|MU_TIMER,
                       0,0,0,
                       0,0,0,0,0,
                       0,0,0,0,0,
                       msg,
                       0,0,
                       &mx,&my,&dummy,
                       &kstate,&kchar,&dummy)&MU_KEYBD);
    }
  }

/* - - end of main loop - - - - - - - - - - - - - - - - - - - -	*/

  emu_exit();

  cpu_free_lines(cpu_handle);

  graf_mouse(M_OFF,NULL);

  if (cpu_handle>=0)
    cpu_close(cpu_handle);

  if (scr_handle>=0)
    scr_close(scr_handle);

  if (prf_handle>=0)
    prf_close(prf_handle);

  menu_bar(menu,0);

  graf_mouse(M_ON,NULL);
  graf_mouse(ARROW,NULL);
  wind_update(END_UPDATE);

  rsrc_free();

  v_clsvwk(vdi_handle);
  appl_exit();
  return 0;
}

/* === End ====================================================	*/
