;****************************************************************
;*  support.s
;*			Support functions on Atari ST
;*
;*  2000-02-29  Bodo Wenzel  Creation
;*  2000-05-04  Bodo Wenzel  Several emulations in one executable
;*  2000-05-06  Bodo Wenzel  External RAM pageable
;*  2000-09-17  Bodo Wenzel  Now nearly full screen emulation
;*  2000-10-23  Bodo Wenzel  Interrupts and timer
;*  2000-11-18  Bodo Wenzel  New screen rendering and
;*                           better joypad emulation
;*  2000-12-10  Bodo Wenzel  Window bug removed, STOP corrected
;*  2001-01-17  Bodo Wenzel  Support for color screens (SLOW!)
;*  2001-01-21  Bodo Wenzel  Joystick emulation via keys
;*  2001-02-04  Bodo Wenzel  New stack management
;*  2001-02-10  Bodo Wenzel  Interrupts for hsync implemented
;*  2001-02-12  Bodo Wenzel  Bug with visibility of objects found
;*  2001-02-24  Bodo Wenzel  Support of more cartridges
;****************************************************************
;-schmutzflag fuer geaendertes, mit doppel noetig
;****************************************************************
;
;  (c)2000 Bodo Wenzel
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation; either version 2 of the License, or
;  (at your option) any later version.
; 
;  This program is distributed in the hope that it will be useful,
;  but WITHOUT ANY WARRANTY; without even the implied warranty of
;  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;  GNU General Public License for more details.
; 
;  You should have received a copy of the GNU General Public License
;  along with this program; if not, write to the Free Software
;  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
;****************************************************************

;NOTES:
;
;* Unlike documented for the real part, this emulation never
;  shows 153/0 in the last line to the cpu, but 255.
;* For speeding up the timings for LCD status modes 2 and 3
;  are shortened.
;real:	152 gap 153/0 line   0   line ... 143   vsync 144 gap ...
;emu:	152 gap 255   line   0   line ... 143   vsync 144 gap ...
;i_ly:	-10 gap 143   line 142   line ...  -1   vsync  -2 gap ...
;stat:	1       1          2/3/0      ... 2/3/0       1       ...
;
;* Unlike the real part the interrupts are just sampled at vsync
;  (for vsync and joypad) and at the end of each line (for
;  LYC, timer, hsync, and serial[currently not emulated]).
;
;* The display is built by different functions, each called for
;  just one line. In general every line function uses
;  preconverted patterns and masks. There are counters to manage
;  the wrapping at borders, too.
;* The objects are drawn at last.
;* If data is written into video relevant memory, conversion
;  takes place immediate. Since memory contents change not as
;  often as screen output is made, this saves time.

;================================================================
;	General constants

	include	"constant.s"

STOP_KEY	equ	$1b	;character in ASCII

;================================================================
;	Exports and imports

	xdef	my_vsync	;void my_vsync(void);
	xref	scr_imgs	;void  *scr_imgs[];
	xref	scr_img		;void  **scr_img;
	xref	scr_base	;char  *scr_base;
	xref	scr_bpl		;int   scr_bpl,
	xref	scr_double	;      scr_double;

	xref	scr_gb2st	;void scr_gb2st(void);

	xref	emu_direction	;unsigned char  emu_direction;
	xref	emu_command	;unsigned char  emu_command;
	xref	emu_quit	;char           emu_quit;

	xref	emu_frames	;long emu_frames;

	xdef	opcode_stop	;void opcode_stop(void);

	xdef	mbc_dummy	;void mbc_dummy(void);
	xdef	mbc1_rom_select	;void mbc1_rom_select(void);
	xdef	mbc1_roml_select;void mbc1_roml_select(void);
	xdef	mbc1_romh_select;void mbc1_romh_select(void);
	xdef	mbc1_ram_select	;void mbc1_ram_select(void);
	xdef	mbc1_mode_select;void mbc1_mode_select(void);
	xdef	mbc1_rom_lsb	;int mbc1_rom_lsb;
	xdef	mbc1_rom_msb	;int mbc1_rom_msb;
	xdef	mbc2_rom_select	;void mbc2_rom_select(void);
	xdef	mbc3_rom_select	;void mbc3_rom_select(void);
	xdef	mbc3_ram_select	;void mbc3_ram_select(void);
	xdef	mbc3_rtc_latch	;void mbc3_rtc_latch(void);
	xdef	mbc5_rom_select	;void mbc5_rom_select(void);
	xdef	mbc5_ram_select	;void mbc5_ram_select(void);
	xref	emu_rom_pages	;unsigned char  *emu_rom_pages[];
	xref	emu_ram_pages	;unsigned char  *emu_ram_pages[];

	xdef	io_wr_lcdc	;void io_wr_lcdc(void);
	xdef	io_rd_stat	;void io_rd_stat(void);
	xdef	io_wr_stat	;void io_wr_stat(void);
	xdef	io_wr_scy	;void io_wr_scy(void);
	xdef	io_wr_scx	;void io_wr_scx(void);
	xdef	io_wr_lyc	;void io_wr_lyc(void);
	xdef	io_wr_dma	;void io_wr_dma(void);
	xdef	io_wr_bgp	;void io_wr_bgp(void);
	xdef	io_wr_obp0	;void io_wr_obp0(void);
	xdef	io_wr_obp1	;void io_wr_obp1(void);
	xdef	io_wr_wy	;void io_wr_wy(void);
	xdef	io_wr_wx	;void io_wr_wx(void);

	xdef	sup_globals	;base address of globals

;================================================================

	text

;================================================================
;	copy one images in the screen window, synchronized to the
;	vertical sync by interrupt
;void my_vsync(void);

MV_REGS		reg	d1-d5

my_vsync:
	move.l	scr_img,a0
	move.l	(a0)+,d0	;next image address
	tst.l	(a0)
	bne.s	mv_goon
	lea	-12(a0),a0	;do it circulating
mv_goon:
	move.l	a0,scr_img
	move.l	d0,a0
	move.l	scr_base,a1
	move.w	scr_bpl,d0	;set up

	if	SCR_WIDTH!=20
	error	"SCR_WIDTH and code not appropriate!"
	endif
	rept	SCR_HEIGHT*8
	movem.l	(a0)+,#MV_REGS
	movem.l	#MV_REGS,(a1)
	add.w	d0,a1		;copy everything
	endm

	rts

;================================================================
;	called first in vsync, this is a line routine, too
;void support(void);

support:
	addq.l	#1,emu_frames

	move.w	#OPS_PER_LINE-1,d6

	add.w	#DIV_PER_LINE,T_DIV(a4)

	move.w	T_D_TIMA(a4),d0
	add.w	d0,T_TIMA(a4)
	bcc.s	s_t_exit
	move.b	T_TMA(a4),d0
s_t_loop:
	add.b	d0,T_TIMA(a4)
	bcs.s	s_t_loop
	bset	#2,INT_IF(a4)
s_t_exit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	check if the user likes to stop
;	and debug extension

	tst.b	emu_quit
	beq	s_goon		;don't stop emulation?
	bpl	s_quit		;no debug mode?

	clr.b	emu_quit
s_get_all:
	move.l	a0,-(sp)
	move.l	a1,-(sp)
	move.l	#$00010002,-(sp)
	trap	#13
	addq.w	#4,sp		;Bconstat(CON)
	move.l	(sp)+,a1
	move.l	(sp)+,a0
	tst.b	d0
	beq	s_goon
	move.l	a0,-(sp)
	move.l	a1,-(sp)
	move.l	#$00020002,-(sp)
	trap	#13
	addq.w	#4,sp		;Bconin(CON)
	move.l	(sp)+,a1
	move.l	(sp)+,a0

	cmp.b	#STOP_KEY,d0
	beq.s	s_quit

	move.l	#$00ff000f,d1
	cmp.b	#'5',d0
	beq.s	s_debug_dir
	cmp.b	#' ',d0
	beq.s	s_debug_cmd

	move.l	#$00feff00,d1
	cmp.b	#'6',d0
	beq.s	s_debug_dir
	cmp.b	#'a',d0
	beq.s	s_debug_cmd
	add.l	d1,d1
	cmp.b	#'4',d0
	beq.s	s_debug_dir
	cmp.b	#'b',d0
	beq.s	s_debug_cmd
	add.l	d1,d1
	cmp.b	#'8',d0
	beq.s	s_debug_dir
	cmp.b	#'c',d0
	beq.s	s_debug_cmd
	add.l	d1,d1
	cmp.b	#'2',d0
	beq.s	s_debug_dir
	cmp.b	#'s',d0
	beq.s	s_debug_cmd
	bra	s_get_all

s_debug_dir:
	or.b	d1,emu_direction
	swap	d1
	and.b	d1,emu_direction
	bra	s_get_all

s_debug_cmd:
	or.b	d1,emu_command
	swap	d1
	and.b	d1,emu_command
	bra	s_get_all

s_quit:
	addq.w	#4,sp		;ignore return to emulation

	move.b	#1,emu_quit
	moveq	#USER_STOP,d0	;return to emulation caller
	rts

s_goon:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	set new joypad values

	move.l	JOYPAD_PTR(a4),a2
	move.b	(a2),-(sp)	;save old value

	move.b	emu_direction,d0
	move.b	d0,JOYPAD_DIR(a4)
	move.b	emu_command,d1
	move.b	d1,JOYPAD_CMD(a4)
	and.b	d1,d0
	move.b	d0,JOYPAD_BOTH(a4)

	move.b	(a2),d0
	not.b	d0
	and.b	(sp)+,d0
	sne	d1		;save high-to-low changes

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	count mbc3 rtc

	lea	mbc3_rtc,a2

	btst	#6,mbc3_rtc_run_c(a2)
	bne.s	s_mbc3_quit	;rtc stopped ?

	subq.b	#1,(a2)
	bcc.s	s_mbc3_quit
	move.b	#VSYNC_PER_SEC-1,(a2)+
	addq.b	#1,(a2)
	cmp.b	#60,(a2)
	bne.s	s_mbc3_quit
	clr.b	(a2)+
	addq.b	#1,(a2)
	cmp.b	#60,(a2)
	bne.s	s_mbc3_quit
	clr.b	(a2)+
	addq.b	#1,(a2)
	cmp.b	#24,(a2)
	bne.s	s_mbc3_quit
	clr.b	(a2)+
	addq.b	#1,(a2)+
	bcc.s	s_mbc3_quit
	bchg	#0,(a2)
	beq.s	s_mbc3_quit
	bset	#7,(a2)		;count values
s_mbc3_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	start a new screen

	move.w	#SCR_YMAX-1-(SCR_HEIGHT*8),LCD_I_LY(a4)
	lea	line_gap(pc),a2
	move.l	a2,V_LCD_LINE+2(a4)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	check if interrupts have to be processed, and
;	return to emulation

	moveq	#$10,d0
	and.b	d1,d0
	move.b	int_mask_sv,d1
	and.b	int_mask_stat,d1
	or.b	d1,d0
	or.b	int_mask_vsync,d0
	or.b	d0,INT_IF(a4)
	jmp	INT_CHECK(a4)

;================================================================
;	Emulation of opcode $10 $00 (STOP)

opcode_stop:
	move.l	scr_imgs,a2
	move.l	scr_imgs+4,a3
	move.w	#SCR_HEIGHT*8*SCR_WIDTH/4-1,d0
	moveq	#0,d1
os_cls:
	move.l	d1,(a2)+
	move.l	d1,(a3)+
	dbra	d0,os_cls	;clear screen

	lea	INT_IM(a4),a2
	move.b	(a2),-(sp)
	move.b	d1,(a2)
	pea	os_abort(pc)	;set up for waiting
os_wait:
	bsr	support
	moveq	#$0f,d0
	and.b	JOYPAD_BOTH(a4),d0
	cmp.b	#$0f,d0
	beq.s	os_wait		;wait for user

	addq.w	#4,sp
	move.b	(sp)+,INT_IM(a4)
	rts			;back to emulation

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	user likes to stop the emulation

os_abort:
	move.b	(sp)+,INT_IM(a4)
	subq.w	#2,a0		;restore state for STOP

	addq.w	#4,sp		;ignore return to emulation
	rts

;================================================================
;	Routines for one line on the screen, different kinds

;----------------------------------------------------------------
;	one line of nothing, just counting LY
;	after the end of vsync prepare screen output

line_gap:
	move.w	LCD_I_LY(a4),d0
	subq.w	#1,d0
	cmp.w	#SCR_YMAX-1-153,d0
	beq.s	lg_new_pic	;vsync finished?

	move.w	d0,LCD_I_LY(a4)
	bra	line_quit_return

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	start a new picture, no need to check for on/off

lg_new_pic:
	move.w	#SCR_YMAX-1+1,LCD_I_LY(a4)

	move.b	LCD_SCY(a4),d6
	add.w	d6,d6
	add.w	d6,d6
	move.b	d6,-(sp)
	and.w	#(MAP_WRAP-1)*MAP_WRAP,d6
	add.w	bg_scx_tile(a5),d6
	add.w	d6,d6
	move.w	d6,bg_map_offset(a5)
	move.w	(sp)+,d6
	and.w	#7*$100*4,d6
	lea	bg_tile,a3
	lea	8*$100*4(a3),a2
	add.w	d6,a3
	move.l	a3,bg_tile_ptr(a5)
	move.l	a2,bg_tile_max(a5)	;set up for background

	moveq	#0,d1
	move.w	d1,wi_map_offset(a5)
	move.b	LCD_WY(a4),d1
	move.w	d1,LCD_I_WY(a4)
	lea	wi_tile,a3
	lea	8*$100*4(a3),a2
	move.l	a3,wi_tile_ptr(a5)
	move.l	a2,wi_tile_max(a5)	;set up for window

	clr.w	ob_y_offset(a5)	;set up for objects

	move.l	line_function(a5),V_LCD_LINE+2(a4)

;*** DEBUG >>>

macro	debug	dat		;max 10 calls!
	move.b	dat,d0
	move.b	d2,60(a2)
	move.b	d2,40(a2)
	move.b	d0,20(a2)
	move.b	d0,(a2)+
	move.b	d1,60(a2)
	move.b	d1,40(a2)
	move.b	d1,20(a2)
	move.b	d1,(a2)+
	move.b	d1,60(a3)
	move.b	d1,40(a3)
	move.b	d0,20(a3)
	move.b	d0,(a3)+
	move.b	d1,60(a3)
	move.b	d1,40(a3)
	move.b	d1,20(a3)
	move.b	d1,(a3)+
	endm

macro	debug_init
	move.l	scr_imgs,a2
	move.l	scr_imgs+4,a3
	moveq	#$24,d1
	move.b	d1,d2
	not.b	d2
	endm

;	debug_init
;	debug	LCD_LCDC(a4)

;*** <<< DEBUG

	tst.w	scr_double
	bmi.s	lg_color
	bne.s	lg_double

	move.l	scr_imgs,pic1(a5)
	move.l	scr_imgs+4,pic2(a5)
	bra.s	lg_exit

lg_double:
	move.l	scr_imgs,a2
	move.l	scr_imgs2,a3
	exg	a2,a3
	move.l	a2,scr_imgs
	move.l	a3,scr_imgs2
	move.l	a3,pic1(a5)
	move.l	scr_imgs+4,a2
	move.l	scr_imgs2+4,a3
	exg	a2,a3
	move.l	a2,scr_imgs+4
	move.l	a2,scr_imgs+8
	move.l	a3,scr_imgs2+4
	move.l	a3,scr_imgs2+8
	move.l	a3,pic2(a5)
	bra.s	lg_exit

lg_color:
	move.l	a0,-(sp)
	move.l	a1,-(sp)
	bsr	scr_gb2st
	move.l	(sp)+,a1
	move.l	(sp)+,a0
	move.l	scr_imgs,pic1(a5)
	move.l	scr_imgs+4,pic2(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	return to emulation

lg_exit:
	move.w	#OPS_PER_LINE-1,d6

	add.w	#DIV_PER_LINE,T_DIV(a4)

	move.w	T_D_TIMA(a4),d0
	add.w	d0,T_TIMA(a4)
	bcc.s	lg_t_exit
	move.b	T_TMA(a4),d0
lg_t_loop:
	add.b	d0,T_TIMA(a4)
	bcs.s	lg_t_loop
	bset	#2,INT_IF(a4)
lg_t_exit:

	move.b	int_lyc_reload,int_lyc_count
	bne.s	lg_lyc_exit
	btst	#6,LCD_STAT(a4)
	beq.s	lg_lyc_exit
	subq.b	#1,int_lyc_count
	move.b	int_mask_stat,d0
	or.b	d0,INT_IF(a4)
lg_lyc_exit:

	jmp	INT_CHECK(a4)

;----------------------------------------------------------------
;	one line of empty screen

;a2	pointer into picture 1
;a3	pointer into picture 2

line_none:
	move.l	pic1(a5),a2
	move.l	pic2(a5),a3

	moveq	#SCR_WIDTH/4-1,d0
	moveq	#0,d1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	erase line

ln_loop:
	move.l	d1,(a2)+
	move.l	d1,(a3)+
	dbra	d0,ln_loop

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update offsets and pointers

	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	LY stuff

line_quit:
	subq.w	#1,LCD_I_LY(a4)
	bcc.s	line_quit_return	;rendering not finished?

	lea	support(pc),a2
	move.l	a2,V_LCD_LINE+2(a4)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	return to emulation

line_quit_return:
	move.w	#OPS_PER_LINE-1,d6

	add.w	#DIV_PER_LINE,T_DIV(a4)

	move.w	T_D_TIMA(a4),d0
	add.w	d0,T_TIMA(a4)
	bcc.s	lqr_t_exit
	move.b	T_TMA(a4),d0
lqr_t_loop:
	add.b	d0,T_TIMA(a4)
	bcs.s	lqr_t_loop
	bset	#2,INT_IF(a4)
lqr_t_exit:

	subq.b	#1,int_lyc_count
	bcc.s	lqr_lyc_exit
	btst	#6,LCD_STAT(a4)
	beq.s	lqr_lyc_exit
	move.b	int_mask_stat,d0
	or.b	d0,INT_IF(a4)
lqr_lyc_exit:

	move.b	int_mask_sh,d0
	and.b	int_mask_stat,d0
	or.b	d0,INT_IF(a4)
	jmp	INT_CHECK(a4)

;----------------------------------------------------------------
;	one line of objects

LO_REGS		reg	a0-a1/a4
;a0	pointer into picture 1 incl. offset
;a1	pointer into picture 2 incl. offset
;a2	pointer into picture 1
;a3	pointer into picture 2
;a4	pointer to pixel line in object
;a6	pointer to ob_tab

line_ob:
	movem.l	#LO_REGS,-(sp)

	move.l	pic1(a5),a2
	move.l	pic2(a5),a3

	moveq	#SCR_WIDTH/4-1,d0
	moveq	#0,d1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	erase line

lo_loop:
	move.l	d1,(a2)+
	move.l	d1,(a3)+
	dbra	d0,lo_loop

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update offsets and pointers

	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects behind background/window

	moveq	#SCR_WIDTH,d1
	sub.w	d1,a2
	sub.w	d1,a3

	lea	ob_tab,a6
	move.w	ob_y_offset(a5),d0
	move.w	(a6,d0.w),d1
	bmi.s	lo_bg_quit	;no objects behind bg/wi?

lo_bg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a4)+,d2
	move.b	(a4)+,d6
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lo_bg_loop	;more objects?
lo_bg_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects in front of background/window

	addq.w	#2,d0
	move.w	(a6,d0.w),d1
	bmi.s	lo_fg_quit	;no objects in front of bg/wi?

lo_fg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a4)+,d2
	move.b	(a4)+,d6
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lo_fg_loop	;more objects?
lo_fg_quit:

	addq.w	#2,d0
	move.w	d0,ob_y_offset(a5)

	movem.l	(sp)+,#LO_REGS
	bra	line_quit

;----------------------------------------------------------------
;	one line of background, with checking for window

line_bc:
	move.w	LCD_I_WY(a4),d0
	dbmi	d0,lbc_bg	;window not reached?
	move.w	d0,LCD_I_WY(a4)

	lea	line_bw(pc),a2
	move.l	a2,V_LCD_LINE+2(a4)
	jmp	(a2)		;now render with window

lbc_bg:
	move.w	d0,LCD_I_WY(a4)

	;falls through to line_bg

;----------------------------------------------------------------
;	one line of background

LB_REGS		reg	a0-a1/a4
;a0	pointer to left tile pattern
;a1	pointer to right tile pattern
;a2	pointer into picture 1
;a3	pointer into picture 2
;a4	pointer to pixel line base
;a6	pointer to actual tile number in background

line_bg:
	movem.l	#LB_REGS,-(sp)

	move.l	pic1(a5),a2
	move.l	pic2(a5),a3
	move.l	bg_tile_ptr(a5),a4
	move.l	bg_map_start(a5),a6
	add.w	bg_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bg_count2(a5),d0
	move.w	bg_count1(a5),d2
	bmi.s	lb_loop2	;no line wrap?
	bra.s	lb_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lb_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lb_next1:
	dbra	d2,lb_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lb_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lb_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update pointers

	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lb_exit		;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lb_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lb_newline:
	move.w	d0,bg_map_offset(a5)

lb_exit:
	move.l	a4,bg_tile_ptr(a5)

	movem.l	(sp)+,#LB_REGS
	bra	line_quit

;----------------------------------------------------------------
;	one line of background and window

LBW_REGS	reg	a0-a1/a4
;a0	pointer to left tile pattern
;a1	pointer to right tile pattern
;a2	pointer into picture 1
;a3	pointer into picture 2
;a4	pointer to pixel line base
;a6	pointer to actual tile number in background

line_bw:
	movem.l	#LBW_REGS,-(sp)

	move.l	pic1(a5),a2
	move.l	pic2(a5),a3
	move.l	bg_tile_ptr(a5),a4

	move.w	bw_count2(a5),d0
	bmi	lbw_wi		;only window?

	move.l	bg_map_start(a5),a6
	add.w	bg_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bw_count1(a5),d2
	bmi.s	lbw_loop2	;no wrap before window?
	bra.s	lbw_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lbw_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbw_next1:
	dbra	d2,lbw_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until window

lbw_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lbw_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update background pointers

	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbw_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbw_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbw_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbw_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window after background

	move.l	wi_tile_ptr(a5),a4

	move.w	bw_count3(a5),d0
	bmi	lbw_exit	;window not visible?

	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.b	wi_wx_mask(a5),d2
	and.b	d2,-(a2)
	and.b	d2,-(a3)

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to first pattern

	move.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,d2
	or.b	d2,(a2)+
	or.b	d1,(a3)+	;store first pattern
	bra.s	lbw_next3

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window only, but first update background pointers

lbw_wi:
	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbw_w_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbw_w_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbw_w_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbw_w_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

	move.l	wi_tile_ptr(a5),a4
	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bw_count3(a5),d0

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lbw_loop3:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbw_next3:
	dbra	d0,lbw_loop3

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update picture and window pointers

lbw_exit:
	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	wi_tile_max(a5),a4
	bcs.s	lbw_wi_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	add.w	#MAP_WRAP*2,wi_map_offset(a5)

lbw_wi_exit:
	move.l	a4,wi_tile_ptr(a5)

	movem.l	(sp)+,#LBW_REGS
	bra	line_quit

;----------------------------------------------------------------
;	one line of backgr. and objects, with checking for window

line_bco:
	move.w	LCD_I_WY(a4),d0
	dbmi	d0,lbco_bg	;window not reached?
	move.w	d0,LCD_I_WY(a4)

	lea	line_bwo(pc),a2
	move.l	a2,V_LCD_LINE+2(a4)
	jmp	(a2)		;now render with window

lbco_bg:
	move.w	d0,LCD_I_WY(a4)

	;falls through to line_bo

;----------------------------------------------------------------
;	one line of background and objects

LBO_REGS	reg	d3/a0-a1/a4
;d3	buffer for a5
;a0	pointer to left tile pattern / picture 1 incl. offset
;a1	pointer to right tile pattern / picture 2 incl. offset
;a2	pointer into picture 1
;a3	pointer into picture 2
;a4	pointer pixel line base / to pixel line in object
;a5	(saved) pointer to mask
;a6	pointer to actual tile number in background / ob_tab

line_bo:
	movem.l	#LBO_REGS,-(sp)

	move.l	pic1(a5),a2
	move.l	pic2(a5),a3
	move.l	bg_tile_ptr(a5),a4
	move.l	bg_map_start(a5),a6
	add.w	bg_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	lea	ob_tab,a1
	move.w	ob_y_offset(a5),d1
	move.w	(a1,d1.w),d1
	bpl	lbo_masking	;objects behind backgr./window?

	move.w	bg_count2(a5),d0
	move.w	bg_count1(a5),d2
	bmi.s	lbo_loop2	;no line wrap?
	bra.s	lbo_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lbo_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbo_next1:
	dbra	d2,lbo_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lbo_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lbo_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update pointers

	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbo_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbo_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbo_newline:
	move.w	d0,bg_map_offset(a5)

lbo_exit:
	move.l	a4,bg_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	prepare for objects

	moveq	#SCR_WIDTH,d1
	sub.w	d1,a2
	sub.w	d1,a3

	lea	ob_tab,a6
	bra	lbo_foreground

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render and save mask for this line

lbo_masking:
	move.w	bg_count2(a5),d0
	move.w	bg_count1(a5),d2
	move.l	a5,d3
	lea	bw_mask(a5),a5

	tst.w	d2
	bmi.s	lbo_m_loop2	;no line wrap?
	bra.s	lbo_m_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lbo_m_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.b	8*$100*4-1(a1),d1
	or.b	8*$100*4(a0),d1
	move.b	d1,(a5)+	;store mask

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbo_m_next1:
	dbra	d2,lbo_m_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lbo_m_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.b	8*$100*4-1(a1),d1
	or.b	8*$100*4(a0),d1
	move.b	d1,(a5)+	;store mask

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lbo_m_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update pointers

	move.l	d3,a5
	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbo_m_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbo_m_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbo_m_newline:
	move.w	d0,bg_map_offset(a5)

lbo_m_exit:
	move.l	a4,bg_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects behind background/window

	moveq	#SCR_WIDTH,d1
	sub.w	d1,a2
	sub.w	d1,a3

	lea	ob_tab,a6
	move.w	ob_y_offset(a5),d0
	move.w	(a6,d0.w),d1
	lea	bw_mask(a5),a5

lbo_bg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a5,d1.w),d0
	move.b	1(a5,d1.w),d1
	move.b	(a4)+,d2
	or.b	d0,d2
	not.b	d0
	move.b	(a4)+,d6
	or.b	d1,d6
	not.b	d1
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	and.b	d0,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	and.b	d0,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	and.b	d1,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	and.b	d1,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lbo_bg_loop	;more objects?

	move.l	d3,a5

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects in front of background/window

lbo_foreground:
	move.w	ob_y_offset(a5),d0
	addq.w	#2,d0

	move.w	(a6,d0.w),d1
	bmi.s	lbo_fg_quit	;no objects in front of bg/wi?

lbo_fg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a4)+,d2
	move.b	(a4)+,d6
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lbo_fg_loop	;more objects?
lbo_fg_quit:

	addq.w	#2,d0
	move.w	d0,ob_y_offset(a5)

	movem.l	(sp)+,#LBO_REGS
	bra	line_quit

;----------------------------------------------------------------
;	one line of background, window, and objects

LBWO_REGS	reg	d3/a0-a1/a4
;d3	buffer for a5
;a0	pointer to left tile pattern / picture 1 incl. offset
;a1	pointer to right tile pattern / picture 2 incl. offset
;a2	pointer into picture 1
;a3	pointer into picture 2
;a4	pointer pixel line base / to pixel line in object
;a5	(saved) pointer to mask
;a6	pointer to actual tile number in background / ob_tab

line_bwo:
	movem.l	#LBWO_REGS,-(sp)

	move.l	pic1(a5),a2
	move.l	pic2(a5),a3
	move.l	bg_tile_ptr(a5),a4

	lea	ob_tab,a1
	move.w	ob_y_offset(a5),d1
	move.w	(a1,d1.w),d1
	bpl	lbwo_masking	;objects behind backgr./window?

	move.w	bw_count2(a5),d0
	bmi	lbwo_wi		;only window?

	move.l	bg_map_start(a5),a6
	add.w	bg_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bw_count1(a5),d2
	bmi.s	lbwo_loop2	;no wrap before window?
	bra.s	lbwo_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lbwo_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbwo_next1:
	dbra	d2,lbwo_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until window

lbwo_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lbwo_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update background pointers

	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbwo_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbwo_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbwo_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbwo_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window after background

	move.l	wi_tile_ptr(a5),a4

	move.w	bw_count3(a5),d0
	bmi	lbwo_exit	;window not visible?

	move.b	wi_wx_mask(a5),d2
	and.b	d2,-(a2)
	and.b	d2,-(a3)

	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to first pattern

	move.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,d2
	or.b	d2,(a2)+
	or.b	d1,(a3)+	;store first pattern
	bra.s	lbwo_next3

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window only, but first update background pointers

lbwo_wi:
	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbwo_w_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbwo_w_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbwo_w_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbwo_w_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

	move.l	wi_tile_ptr(a5),a4
	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bw_count3(a5),d0

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lbwo_loop3:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbwo_next3:
	dbra	d0,lbwo_loop3

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update picture and window pointers

lbwo_exit:
	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	wi_tile_max(a5),a4
	bcs.s	lbwo_wi_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	add.w	#MAP_WRAP*2,wi_map_offset(a5)

lbwo_wi_exit:
	move.l	a4,wi_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	prepare for objects

	moveq	#SCR_WIDTH,d1
	sub.w	d1,a2
	sub.w	d1,a3

	lea	ob_tab,a6
	bra	lbwo_foreground

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render and save mask for this line

lbwo_masking:
	move.w	bw_count2(a5),d0
	bmi	lbwo_m_wi	;only window?

	move.l	bg_map_start(a5),a6
	add.w	bg_map_offset(a5),a6

	move.w	bw_count1(a5),d2
	move.l	a5,d3
	lea	bw_mask(a5),a5

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	tst.w	d2
	bmi.s	lbwo_m_loop2	;no wrap before window?
	bra.s	lbwo_m_next1

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until wrap

lbwo_m_loop1:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.b	8*$100*4-1(a1),d1
	or.b	8*$100*4(a0),d1
	move.b	d1,(a5)+	;store mask

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbwo_m_next1:
	dbra	d2,lbwo_m_loop1

	lea	-MAP_WRAP*2(a6),a6	;wrap pointer to tile nr

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until window

lbwo_m_loop2:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.b	8*$100*4-1(a1),d1
	or.b	8*$100*4(a0),d1
	move.b	d1,(a5)+	;store mask

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns

	dbra	d0,lbwo_m_loop2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update background pointers

	exg	d3,a5
	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbwo_m_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbwo_m_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbwo_m_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbwo_m_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window after background

	move.l	wi_tile_ptr(a5),a4

	move.w	bw_count3(a5),d0
	bmi	lbwo_m_exit	;window not visible?

	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.b	wi_wx_mask(a5),d2
	exg	d3,a5
	and.b	d2,-(a2)
	and.b	d2,-(a3)
	and.b	d2,-(a5)

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to first pattern

	move.b	8*$100*4(a0),d1
	or.b	d1,(a5)+	;store mask

	move.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,d2
	or.b	d2,(a2)+
	or.b	d1,(a3)+	;store first pattern
	bra.s	lbwo_m_next3

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render window only, but first update background pointers

lbwo_m_wi:
	lea	$100*4(a4),a4
	cmp.l	bg_tile_max(a5),a4
	bcs.s	lbwo_m_w_bg_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	moveq	#MAP_WRAP*2,d0
	add.w	bg_map_offset(a5),d0
	cmp.w	#MAP_WRAP*MAP_WRAP*2,d0
	bcs.s	lbwo_m_w_bg_newline	;no vertical wrap?
	sub.w	#MAP_WRAP*MAP_WRAP*2,d0
lbwo_m_w_bg_newline:
	move.w	d0,bg_map_offset(a5)

lbwo_m_w_bg_exit:
	move.l	a4,bg_tile_ptr(a5)

	move.l	wi_tile_ptr(a5),a4
	move.l	wi_map_start(a5),a6
	add.w	wi_map_offset(a5),a6

	move.w	(a6)+,d1
	lea	2(a4,d1.w),a0	;pointer to "old" right pattern

	move.w	bw_count3(a5),d0
	move.l	a5,d3
	lea	bw_mask(a5),a5

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	render until finished

lbwo_m_loop3:
	move.l	a0,a1		;right becomes left

	move.w	(a6)+,d1
	lea	(a4,d1.w),a0	;pointer to right pattern

	move.b	8*$100*4-1(a1),d1
	or.b	8*$100*4(a0),d1
	move.b	d1,(a5)+	;store mask

	move.w	(a1)+,d1
	or.w	(a0)+,d1
	move.w	d1,-(sp)
	move.b	(sp)+,(a2)+
	move.b	d1,(a3)+	;join patterns
lbwo_m_next3:
	dbra	d0,lbwo_m_loop3

	move.l	d3,a5

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	update picture and window pointers

lbwo_m_exit:
	move.l	a2,pic1(a5)
	move.l	a3,pic2(a5)

	lea	$100*4(a4),a4
	cmp.l	wi_tile_max(a5),a4
	bcs.s	lbwo_m_wi_exit	;not last pixel line?

	lea	-8*$100*4(a4),a4

	add.w	#MAP_WRAP*2,wi_map_offset(a5)

lbwo_m_wi_exit:
	move.l	a4,wi_tile_ptr(a5)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects behind background/window

	moveq	#SCR_WIDTH,d1
	sub.w	d1,a2
	sub.w	d1,a3

	lea	ob_tab,a6
	move.w	ob_y_offset(a5),d0
	move.w	(a6,d0.w),d1
	lea	bw_mask(a5),a5

lbwo_bg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a5,d1.w),d0
	move.b	1(a5,d1.w),d1
	move.b	(a4)+,d2
	or.b	d0,d2
	not.b	d0
	move.b	(a4)+,d6
	or.b	d1,d6
	not.b	d1
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	and.b	d0,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	and.b	d0,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	and.b	d1,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	and.b	d1,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lbwo_bg_loop	;more objects?

	move.l	d3,a5

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	objects in front of background/window

lbwo_foreground:
	move.w	ob_y_offset(a5),d0
	addq.w	#2,d0

	move.w	(a6,d0.w),d1
	bmi.s	lbwo_fg_quit	;no objects in front of bg/wi?

lbwo_fg_loop:
	lea	(a6,d1.w),a4
	move.w	(a4)+,d1
	lea	(a2,d1.w),a0
	lea	(a3,d1.w),a1
	move.b	(a4)+,d2
	move.b	(a4)+,d6
	and.b	d2,(a0)
	and.b	d2,(a1)
	move.b	(a4)+,d2
	or.b	d2,(a0)+
	move.b	(a4)+,d2
	or.b	d2,(a1)+
	and.b	d6,(a0)
	and.b	d6,(a1)
	move.b	(a4)+,d6
	or.b	d6,(a0)
	move.b	(a4)+,d6
	or.b	d6,(a1)		;insert object

	move.w	(a4),d1
	bpl.s	lbwo_fg_loop	;more objects?
lbwo_fg_quit:

	addq.w	#2,d0
	move.w	d0,ob_y_offset(a5)

	movem.l	(sp)+,#LBWO_REGS
	bra	line_quit

;================================================================
;	conversion function, preparing GB data for ST screen

;----------------------------------------------------------------
;	color conversions for all cases

color_tab:
	dc.w	color_0000-color_tab-0
	dc.w	color_0001-color_tab-2
	dc.w	color_0010-color_tab-4
	dc.w	color_0011-color_tab-6
	dc.w	color_0100-color_tab-8
	dc.w	color_0101-color_tab-10
	dc.w	color_0110-color_tab-12
	dc.w	color_0111-color_tab-14
	dc.w	color_1000-color_tab-16
	dc.w	color_1001-color_tab-18
	dc.w	color_1010-color_tab-20
	dc.w	color_1011-color_tab-22
	dc.w	color_1100-color_tab-24
	dc.w	color_1101-color_tab-26
	dc.w	color_1110-color_tab-28
	dc.w	color_1111-color_tab-30

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0000:
	sub.b	d0,d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0001:
	move.b	1(a3),d0
	or.b	(a3),d0
	not.b	d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0010:
	move.b	1(a3),d0
	not.b	d0
	and.b	(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0011:
	move.b	1(a3),d0
	not.b	d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0100:
	move.b	(a3),d0
	not.b	d0
	and.b	1(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0101:
	move.b	(a3),d0
	not.b	d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0110:
	move.b	1(a3),d0
	move.b	(a3),d1
	eor.b	d1,d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_0111:
	move.b	1(a3),d0
	and.b	(a3),d0
	not.b	d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1000:
	move.b	1(a3),d0
	and.b	(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1001:
	move.b	1(a3),d0
	move.b	(a3),d1
	eor.b	d1,d0
	not.b	d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1010:
	move.b	(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1011:
	move.b	1(a3),d0
	not.b	d0
	or.b	(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1100:
	move.b	1(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1101:
	move.b	(a3),d0
	not.b	d0
	or.b	1(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1110:
	move.b	1(a3),d0
	or.b	(a3),d0
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

color_1111:
	move.b	#$ff,d0
	rts

;----------------------------------------------------------------
;	convert background or window tile

;d0	(pattern)
;d1	(mask)
;d2	shift left 0..7 pixel
;d3.w	(counter)
;a0,a1	addresses of color functions
;a2	pointer to destination, will be increased by 4
;a3	pointer to source, will be increased by 8*2

convert_bw:
	move.w	#8-1,d3
cbw_loop:
	moveq	#0,d0
	jsr	(a0)
	swap	d0
	jsr	(a1)
	lsl.l	d2,d0
	movep.w	d0,1(a2)
	swap	d0
	movep.w	d0,0(a2)
	moveq	#0,d1
	move.b	(a3)+,d1
	or.b	(a3)+,d1
	lsl.w	d2,d1
	move.w	d1,8*$100*4(a2)

	lea	$100*4(a2),a2

	dbra	d3,cbw_loop
	lea	4-8*$100*4(a2),a2
	rts

;----------------------------------------------------------------
;	convert all marked background or window tiles

convert_bw_all_marked:
	move.w	d0,-(sp)
	move.w	d1,-(sp)

	move.w	#$80-1,d3
	move.l	bw_tile_low(a5),a3
cbwam_l_loop:
	move.b	(a6)+,d0
	bmi.s	cbwam_l_next
	cmp.b	d0,d2
	bne.s	cbwam_l_conv	;conversion necessary?
cbwam_l_next:
	addq.w	#4,a2
	lea	2*8(a3),a3
	dbra	d3,cbwam_l_loop
	bra.s	cbwam_l_ready
cbwam_l_conv:
	move.b	d2,-1(a6)
	swap	d3
	bsr	convert_bw
	swap	d3
	dbra	d3,cbwam_l_loop
cbwam_l_ready:			;convert tiles $00..$7f

	move.w	#$80-1,d3
	move.l	bw_tile_high(a5),a3
	lea	$800(a3),a3
cbwam_h_loop:
	move.b	(a6)+,d0
	bmi.s	cbwam_h_next
	cmp.b	d0,d2
	bne.s	cbwam_h_conv	;conversion necessary?
cbwam_h_next:
	addq.w	#4,a2
	lea	2*8(a3),a3
	dbra	d3,cbwam_h_loop
	bra.s	cbwam_h_ready
cbwam_h_conv:
	move.b	d2,-1(a6)
	swap	d3
	bsr	convert_bw
	swap	d3
	dbra	d3,cbwam_h_loop
cbwam_h_ready:			;convert tiles $80..$ff
	move.w	(sp)+,d1
	move.w	(sp)+,d0
	rts

;----------------------------------------------------------------
;	convert all used background or window tiles

convert_bw_all_used:
	move.w	#$80-1,d3
	move.l	bw_tile_low(a5),a3
cbwau_l_loop:
	move.b	(a6)+,d0
	bmi.s	cbwau_l_next
	cmp.b	d0,d2
	beq.s	cbwau_l_conv	;conversion necessary?
cbwau_l_next:
	addq.w	#4,a2
	lea	2*8(a3),a3
	dbra	d3,cbwau_l_loop
	bra.s	cbwau_l_ready
cbwau_l_conv:
	move.b	d2,-1(a6)
	swap	d3
	bsr	convert_bw
	swap	d3
	dbra	d3,cbwau_l_loop
cbwau_l_ready:			;convert tiles $00..$7f

	move.w	#$80-1,d3
	move.l	bw_tile_high(a5),a3
	lea	$800(a3),a3
cbwau_h_loop:
	move.b	(a6)+,d0
	bmi.s	cbwau_h_next
	cmp.b	d0,d2
	beq.s	cbwau_h_conv	;conversion necessary?
cbwau_h_next:
	addq.w	#4,a2
	lea	2*8(a3),a3
	dbra	d3,cbwau_h_loop
	bra.s	cbwau_h_ready
cbwau_h_conv:
	move.b	d2,-1(a6)
	swap	d3
	bsr	convert_bw
	swap	d3
	dbra	d3,cbwau_h_loop
cbwau_h_ready:			;convert tiles $80..$ff
	rts

;----------------------------------------------------------------
;	table of reversed bytes for object conversion

revers_tab:
	dc.b	$00,$80,$40,$c0,$20,$a0,$60,$e0
	dc.b	$10,$90,$50,$d0,$30,$b0,$70,$f0
	dc.b	$08,$88,$48,$c8,$28,$a8,$68,$e8
	dc.b	$18,$98,$58,$d8,$38,$b8,$78,$f8
	dc.b	$04,$84,$44,$c4,$24,$a4,$64,$e4
	dc.b	$14,$94,$54,$d4,$34,$b4,$74,$f4
	dc.b	$0c,$8c,$4c,$cc,$2c,$ac,$6c,$ec
	dc.b	$1c,$9c,$5c,$dc,$3c,$bc,$7c,$fc
	dc.b	$02,$82,$42,$c2,$22,$a2,$62,$e2
	dc.b	$12,$92,$52,$d2,$32,$b2,$72,$f2
	dc.b	$0a,$8a,$4a,$ca,$2a,$aa,$6a,$ea
	dc.b	$1a,$9a,$5a,$da,$3a,$ba,$7a,$fa
	dc.b	$06,$86,$46,$c6,$26,$a6,$66,$e6
	dc.b	$16,$96,$56,$d6,$36,$b6,$76,$f6
	dc.b	$0e,$8e,$4e,$ce,$2e,$ae,$6e,$ee
	dc.b	$1e,$9e,$5e,$de,$3e,$be,$7e,$fe
	dc.b	$01,$81,$41,$c1,$21,$a1,$61,$e1
	dc.b	$11,$91,$51,$d1,$31,$b1,$71,$f1
	dc.b	$09,$89,$49,$c9,$29,$a9,$69,$e9
	dc.b	$19,$99,$59,$d9,$39,$b9,$79,$f9
	dc.b	$05,$85,$45,$c5,$25,$a5,$65,$e5
	dc.b	$15,$95,$55,$d5,$35,$b5,$75,$f5
	dc.b	$0d,$8d,$4d,$cd,$2d,$ad,$6d,$ed
	dc.b	$1d,$9d,$5d,$dd,$3d,$bd,$7d,$fd
	dc.b	$03,$83,$43,$c3,$23,$a3,$63,$e3
	dc.b	$13,$93,$53,$d3,$33,$b3,$73,$f3
	dc.b	$0b,$8b,$4b,$cb,$2b,$ab,$6b,$eb
	dc.b	$1b,$9b,$5b,$db,$3b,$bb,$7b,$fb
	dc.b	$07,$87,$47,$c7,$27,$a7,$67,$e7
	dc.b	$17,$97,$57,$d7,$37,$b7,$77,$f7
	dc.b	$0f,$8f,$4f,$cf,$2f,$af,$6f,$ef
	dc.b	$1f,$9f,$5f,$df,$3f,$bf,$7f,$ff

;----------------------------------------------------------------
;	convert object tile

;d0	object flags / (shifted pattern)
;d1	(shifted mask)
;d2	(shift left/right 0..7 pixel)
;d3.w	counter 16-1 / 8-1 / 1-1
;d4	object x+8 / (screen offset)
;d5	(mask)
;a0,a1	addresses of color functions
;a2	pointer to destination, may be increased
;a3	pointer to source, may be increased
;a6	(pointer to revers_tab)

convert_ob:
	tst.b	d4
	beq.s	co_quit
	cmp.b	#SCR_WIDTH*8+8,d4
	bcc.s	co_quit
	moveq	#0,d5
	moveq	#7,d2
	cmp.b	#SCR_WIDTH*8,d4
	bcc	co_right
	sub.b	#9,d4
	bcc.s	co_mid
	not.b	d4
	and.b	d4,d2
	addq.b	#8,d2
	moveq	#0,d4
	bra.s	co_left
co_quit:
	rts

co_mid:
	eor.b	d2,d4
	and.b	d4,d2
	lsr.b	#3,d4
	ext.w	d4

co_left:
	moveq	#$40,d1
	and.b	d0,d1
	beq.s	co_lxn
	moveq	#$20,d1
	and.b	d0,d1
	bne	co_lff
	bra.s	co_lnf
co_lxn:
	moveq	#$20,d1
	and.b	d0,d1
	bne.s	co_lfn

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting left x normal, y normal

co_lnn_loop:
	move.w	d4,(a2)+
	move.b	(a3),d5
	or.b	1(a3),d5
	moveq	#0,d0
	jsr	(a0)
	and.b	d5,d0
	lsl.w	d2,d0
	movep.w	d0,2(a2)
	moveq	#0,d0
	jsr	(a1)
	and.b	d5,d0
	lsl.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	lsl.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#2,a3
	addq.w	#8,a2
	dbra	d3,co_lnn_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting left x normal, y flipped

co_lnf:
	move.w	d3,d0
	addq.w	#1,d0
	add.w	d0,d0
	add.w	d0,a3
co_lnf_loop:
	move.w	d4,(a2)+
	move.b	-(a3),d5
	or.b	-(a3),d5
	moveq	#0,d0
	jsr	(a0)
	and.b	d5,d0
	lsl.w	d2,d0
	movep.w	d0,2(a2)
	moveq	#0,d0
	jsr	(a1)
	and.b	d5,d0
	lsl.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	lsl.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#8,a2
	dbra	d3,co_lnf_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting left x flipped, y normal

co_lfn:
	lea	revers_tab(pc),a6
co_lfn_loop:
	move.w	d4,(a2)+
	move.b	(a3),d5
	or.b	1(a3),d5
	moveq	#0,d0
	jsr	(a0)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsl.w	d2,d0
	movep.w	d0,2(a2)
	moveq	#0,d0
	jsr	(a1)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsl.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	move.b	(a6,d1.w),d1
	lsl.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#2,a3
	addq.w	#8,a2
	dbra	d3,co_lfn_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting left x flipped, y flipped

co_lff:
	move.w	d3,d0
	addq.w	#1,d0
	add.w	d0,d0
	add.w	d0,a3
	lea	revers_tab(pc),a6
co_lff_loop:
	move.w	d4,(a2)+
	move.b	-(a3),d5
	or.b	-(a3),d5
	moveq	#0,d0
	jsr	(a0)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsl.w	d2,d0
	movep.w	d0,2(a2)
	moveq	#0,d0
	jsr	(a1)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsl.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	move.b	(a6,d1.w),d1
	lsl.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#8,a2
	dbra	d3,co_lff_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting right

co_right:
	and.b	d4,d2
	moveq	#SCR_WIDTH-2,d4

	moveq	#$40,d1
	and.b	d0,d1
	beq.s	co_rxn
	moveq	#$20,d1
	and.b	d0,d1
	bne	co_rff
	bra.s	co_rnf
co_rxn:
	moveq	#$20,d1
	and.b	d0,d1
	bne.s	co_rfn

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting right x normal, y normal

	moveq	#0,d0
co_rnn_loop:
	move.w	d4,(a2)+
	move.b	(a3),d5
	or.b	1(a3),d5
	jsr	(a0)
	and.b	d5,d0
	lsr.w	d2,d0
	movep.w	d0,2(a2)
	jsr	(a1)
	and.b	d5,d0
	lsr.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	lsr.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#2,a3
	addq.w	#8,a2
	dbra	d3,co_rnn_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting right x normal, y flipped

co_rnf:
	move.w	d3,d0
	addq.w	#1,d0
	add.w	d0,d0
	add.w	d0,a3
	moveq	#0,d0
co_rnf_loop:
	move.w	d4,(a2)+
	move.b	-(a3),d5
	or.b	-(a3),d5
	jsr	(a0)
	and.b	d5,d0
	lsr.w	d2,d0
	movep.w	d0,2(a2)
	jsr	(a1)
	and.b	d5,d0
	lsr.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	lsr.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#8,a2
	dbra	d3,co_rnf_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting right x flipped, y normal

co_rfn:
	lea	revers_tab(pc),a6
	moveq	#0,d0
co_rfn_loop:
	move.w	d4,(a2)+
	move.b	(a3),d5
	or.b	1(a3),d5
	jsr	(a0)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsr.w	d2,d0
	movep.w	d0,2(a2)
	jsr	(a1)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsr.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	move.b	(a6,d1.w),d1
	lsr.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#2,a3
	addq.w	#8,a2
	dbra	d3,co_rfn_loop
	rts

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert shifting right x flipped, y flipped

co_rff:
	move.w	d3,d0
	addq.w	#1,d0
	add.w	d0,d0
	add.w	d0,a3
	lea	revers_tab(pc),a6
	moveq	#0,d0
co_rff_loop:
	move.w	d4,(a2)+
	move.b	-(a3),d5
	or.b	-(a3),d5
	jsr	(a0)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsr.w	d2,d0
	movep.w	d0,2(a2)
	jsr	(a1)
	and.b	d5,d0
	move.b	(a6,d0.w),d0
	lsr.w	d2,d0
	movep.w	d0,3(a2)
	move.w	d5,d1
	move.b	(a6,d1.w),d1
	lsr.w	d2,d1
	not.w	d1
	move.w	d1,(a2)+
	addq.w	#8,a2
	dbra	d3,co_rff_loop
	rts

;----------------------------------------------------------------
;	link an object (two entries, the 2nd for initialization)

;d0	offset to object in ob_tab
;d1	(scratch)
;d2	(scratch)
;d3.w	(counter 16-1 / 8-1)
;d4.w	(scratch)
;a3	pointer into oam
;a6	pointer to ob_tab

ob_link:
	move.w	d0,d4
	addq.w	#4*2,d4
	move.w	ob_count(a5),d3
ol_out_loop:
	move.w	(a6,d4.w),d1
	move.w	2(a6,d4.w),d2
	move.w	d1,(a6,d2.w)
	move.w	d2,10(a6,d1.w)
	add.w	#6*2,d4
	dbra	d3,ol_out_loop

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	check for visibility

ob_link_in:
	move.w	d0,d4
	move.w	ob_count(a5),d3

	moveq	#0,d1
	move.b	(a3),d1
	beq.s	ol_off
	move.b	1(a3),d2
	beq.s	ol_off
	cmp.b	#SCR_WIDTH*8+8,d2
	bcc.s	ol_off
	sub.w	#16,d1
	bcs.s	ol_topclip
	move.w	#SCR_HEIGHT*8-1,d2
	sub.w	d1,d2
	bcs.s	ol_off
	sub.w	d3,d2
	bcc.s	ol_goon
	add.w	d2,d3
	not.w	d2
	move.w	d2,-(sp)
	bsr.s	ol_goon
	move.w	(sp)+,d3

ol_off:
	addq.w	#4*2,d4
	moveq	#-6*2,d2
ol_o_loop:
	move.w	d2,(a6,d4.w)
	move.w	d4,2(a6,d4.w)
	sub.w	d2,d4
	dbra	d3,ol_o_loop
	rts

ol_topclip:
	addq.w	#4*2,d4
	add.w	d1,d3
	not.w	d1
	moveq	#-6*2,d2
ol_t_loop:
	move.w	d2,(a6,d4.w)
	move.w	d4,2(a6,d4.w)
	sub.w	d2,d4
	dbra	d1,ol_t_loop

	tst.w	d3
	bmi.s	ol_quit
	moveq	#0,d1
	subq.w	#4*2,d4

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	now link lines

ol_goon:
	swap	d0

	add.w	d1,d1
	add.w	d1,d1
	moveq	#$80-$100,d2
	and.b	3(a3),d2
	bne.s	ol_l_loop1
	addq.w	#2,d1		;offset background / foreground

ol_l_loop1:
	move.w	d1,d0
ol_l_loop2:
	move.w	(a6,d0.w),d2
	cmp.w	d2,d4
	bge.s	ol_l_insert
	move.w	d2,d0
	addq.w	#4*2,d0
	bra.s	ol_l_loop2

ol_l_insert:
	move.w	d4,(a6,d0.w)
	addq.w	#4*2,d4
	move.w	d2,(a6,d4.w)
	move.w	d0,2(a6,d4.w)
	add.w	#5*2,d2
	move.w	d4,(a6,d2.w)
	addq.w	#2*2,d4
	addq.w	#2*2,d1
	dbra	d3,ol_l_loop1

	swap	d0
ol_quit:
	rts

;----------------------------------------------------------------
;	check if any object uses this tile and convert

OTC_REGS	reg	d1-d5/a0-a3/a6
;d0	(scratch)
;d1.w	offset into video ram
;d2	(scratch)
;d3.w	(scratch)
;a2	base address of video ram
;a3	(scratch)

ob_tile_check:
	move.w	d1,d2
	lsr.w	#4,d2
	swap	d1
	move.w	ob_tilemask(a5),d1
	move.l	BASE_RAM_OAM(a4),a3
	lea	$fe00+2(a3),a3
	move.w	#40-1,d3
otc_loop:
	move.b	(a3),d0
	eor.b	d2,d0
	and.w	d1,d0
	bne.s	otc_next

	movem.l	#OTC_REGS,-(sp)

	move.w	(a3),d0
	move.w	-(a3),d4
	beq.s	otc_not_visible

	move.w	d1,d2
	ext.w	d2
	lsl.w	#4,d2
	not.w	d2
	swap	d1
	and.w	#$fffe,d1
	and.w	d1,d2
	lea	(a2,d1.w),a3
	lea	ob_tab+SCR_HEIGHT*8*2*2,a2
	moveq	#40-1,d5
	sub.w	d3,d5
	lsl.w	#4+1,d5
	add.w	d5,d2
	add.w	d2,d2
	move.w	d2,d1
	add.w	d2,d2
	add.w	d1,d2
	add.w	d2,a2
	moveq	#$10,d2
	and.b	d0,d2
	beq.s	otc_pal0
	move.l	ob_color1_l(a5),a0
	move.l	ob_color1_h(a5),a1
	bra.s	otc_conv
otc_pal0:
	move.l	ob_color0_l(a5),a0
	move.l	ob_color0_h(a5),a1
otc_conv:
	moveq	#1-1,d3
	bsr	convert_ob	;convert one line
otc_not_visible:
	movem.l	(sp)+,#OTC_REGS
otc_next:
	addq.w	#4,a3
	dbra	d3,otc_loop
	swap	d1
	rts

;----------------------------------------------------------------
;	link and convert from scratch

;d0-d5/a0-a3/a6	(scratch, not saved)

ob_link_convert_all:
	move.l	#((-6*2)<<16)+$10000-6*2,d2
	lea	ob_tab,a2
	moveq	#SCR_HEIGHT*8*2/4/2-1,d3
olca_clear_loop:
	move.l	d2,(a2)+
	move.l	d2,(a2)+
	move.l	d2,(a2)+
	move.l	d2,(a2)+
	dbra	d3,olca_clear_loop	;clear line offsets

	move.l	BASE_RAM_OAM(a4),a3
	lea	$fe00+40*4(a3),a3
	lea	ob_tab,a6
	move.w	#SCR_HEIGHT*8*2*2+40*16*6*2,d0
	moveq	#40-1,d5
olca_link_loop:
	subq.w	#4,a3
	sub.w	#16*6*2,d0
	bsr	ob_link_in
	dbra	d5,olca_link_loop	;link in all objects

	lea	ob_tab+SCR_HEIGHT*8*2*2,a2
	move.l	BASE_RAM_VID(a4),a3
	lea	$8000(a3),a3
	move.l	BASE_RAM_OAM(a4),a6
	lea	$fe00(a6),a6
	moveq	#40-1,d3
olca_conv_loop:
	move.w	(a6)+,d4
	move.w	ob_tilemask(a5),d1
	and.b	(a6)+,d1
	move.b	(a6)+,d0
	moveq	#$10,d2
	and.b	d0,d2
	beq.s	olca_pal0
	move.l	ob_color1_l(a5),a0
	move.l	ob_color1_h(a5),a1
	bra.s	olca_conv
olca_pal0:
	move.l	ob_color0_l(a5),a0
	move.l	ob_color0_h(a5),a1
olca_conv:
	swap	d3
	movem.l	a2/a3/a6,-(sp)
	lsl.w	#3+1,d1
	add.w	d1,a3
	move.w	ob_count(a5),d3
	bsr	convert_ob
	movem.l	(sp)+,a2/a3/a6
	swap	d3
	lea	16*6*2(a2),a2
	dbra	d3,olca_conv_loop	;convert all objects
	rts

;================================================================
;	memory emulation

;----------------------------------------------------------------
;	dummy rom write

mbc_dummy:
	jmp	(a6)

;----------------------------------------------------------------
;	switch rom bank (MBC1 4/32 mode)

mbc1_rom_select:
	and.w	#$1f,d0		;32 banks, max. 512 Kbyte
	add.w	d0,d0
	add.w	d0,d0
	move.w	d0,mbc1_rom_lsb
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch rom bank LSB (MBC1 16/8 mode)

mbc1_roml_select:
	and.w	#$1f,d0		;128 banks, max. 2 Mbyte
	add.w	d0,d0
	add.w	d0,d0
	move.w	d0,mbc1_rom_lsb
	move.w	mbc1_rom_msb,d1
	lsl.w	#5,d1
	add.w	d1,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch rom bank MSB (MBC1 16/8 mode)

mbc1_romh_select:
	and.w	#$03,d0		;128 banks, max. 2 Mbyte
	add.w	d0,d0
	add.w	d0,d0
	move.w	d0,mbc1_rom_msb
	lsl.w	#5,d0
	add.w	mbc1_rom_lsb,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch ram bank (MBC1 4/32 mode)

mbc1_ram_select:
	and.w	#$03,d0		;4 banks, max. 32 KByte
	add.w	d0,d0
	add.w	d0,d0
	move.w	d0,mbc1_rom_msb
	lea	emu_ram_pages,a3
	move.l	(a3,d0.w),BASE_RAM_EXT(a4)
	jmp	SP_RAM_CORRECT(a4)

;----------------------------------------------------------------
;	switch mode select (MBC1)

mbc1_mode_select:
	btst	#0,d0
	beq.s	mbc1_mode_16_8

	move.w	mbc1_rom_lsb,d0	;32 banks, max. 512 Kbyte
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)

	move.w	mbc1_rom_msb,d0	;4 banks, max. 32 KByte
	lea	emu_ram_pages,a3
	move.l	(a3,d0.w),BASE_RAM_EXT(a4)

	lea	mbc1_rom_select(pc),a3
	move.l	a3,V_ROML_SELECT+2(a4)
	move.l	a3,V_ROMH_SELECT+2(a4)
	lea	mbc1_ram_select(pc),a3
	move.l	a3,V_RAM_SELECT+2(a4)
	jmp	SP_ROM_CORRECT(a4)

mbc1_mode_16_8:
	move.w	mbc1_rom_msb,d0
	lsl.w	#5,d0		;128 banks, max. 2 Mbyte
	add.w	mbc1_rom_lsb,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)

	move.l	emu_ram_pages,BASE_RAM_EXT(a4)

	lea	mbc1_roml_select(pc),a3
	move.l	a3,V_ROML_SELECT+2(a4)
	move.l	a3,V_ROMH_SELECT+2(a4)
	lea	mbc1_romh_select(pc),a3
	move.l	a3,V_RAM_SELECT+2(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch rom bank (MBC2)

mbc2_rom_select:
	and.w	#$0f,d0		;16 banks, max. 256 Kbyte
	add.w	d0,d0
	add.w	d0,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch rom bank (MBC3)

mbc3_rom_select:
	and.w	#$7f,d0		;128 banks, max. 2 Mbyte
	add.w	d0,d0
	add.w	d0,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch ram bank and rtc register (MBC3)

mbc3_ram_select:
	btst	#3,d0
	bne.s	mbc3_rtc_select

	and.w	#$03,d0		;4 banks, max. 32 KByte
	add.w	d0,d0
	add.w	d0,d0
	lea	emu_ram_pages,a3
	move.l	(a3,d0.w),BASE_RAM_EXT(a4)
	lea	RAM_RD_EXT_STD(a4),a3
	move.l	a3,V_RAM_RD_EXT+2(a4)
	lea	RAM_WR_EXT_STD(a4),a3
	move.l	a3,V_RAM_WR_EXT+2(a4)
	jmp	SP_RAM_CORRECT(a4)

mbc3_rtc_select:
	lea	mbc3_rtc,a3
	lea	mbc3_rtc_lat(a3),a3
	and.w	#$07,d0
	add.w	d0,a3
	move.l	a3,mbc3_rtc_ptr
	lea	mbc3_rtc_rd(pc),a3
	move.l	a3,V_RAM_RD_EXT+2(a4)
	lea	mbc3_rtc_wr(pc),a3
	move.l	a3,V_RAM_WR_EXT+2(a4)
	jmp	(a6)

mbc3_rtc_rd:
	move.l	mbc3_rtc_ptr,a3
	move.l	a3,a2
	jmp	(a6)

mbc3_rtc_wr:
	move.l	mbc3_rtc_ptr,a3
	move.b	d0,mbc3_rtc_run_s-mbc3_rtc_lat(a3)
	jmp	(a6)

;----------------------------------------------------------------
;	latch rtc register (MBC3)

mbc3_rtc_latch:
	btst	#0,d0
	beq.s	mbc3_rtc_quit

	lea	mbc3_rtc,a3
	lea	mbc3_rtc_run_s(a3),a2
	lea	mbc3_rtc_lat(a3),a3
	move.b	(a2)+,(a3)+
	move.b	(a2)+,(a3)+
	move.b	(a2)+,(a3)+
	move.b	(a2)+,(a3)+
	move.b	(a2)+,(a3)+
mbc3_rtc_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	switch rom bank (MBC5)

mbc5_rom_select:
	and.w	#$ff,d0		;256 banks, max. 4 Mbyte
	add.w	d0,d0
	add.w	d0,d0
	lea	emu_rom_pages,a3
	move.l	(a3,d0.w),BASE_ROM_1(a4)
	jmp	SP_ROM_CORRECT(a4)

;----------------------------------------------------------------
;	switch ram bank (MBC5)

mbc5_ram_select:
	and.w	#$03,d0		;4 banks, max. 32 KByte
	add.w	d0,d0
	add.w	d0,d0
	lea	emu_ram_pages,a3
	move.l	(a3,d0.w),BASE_RAM_EXT(a4)
	jmp	SP_RAM_CORRECT(a4)

;----------------------------------------------------------------
;	write without watching

ram_none:
	move.b	d0,(a2,d1.w)	;just store byte in memory
	jmp	(a6)

;----------------------------------------------------------------
;	write into tile table for background

ram_tile_bg:
	cmp.b	(a2,d1.w),d0
	beq.s	rtb_quit	;no change?

	move.w	d3,-(sp)

	move.b	d0,(a2,d1.w)	;store pattern in tile table
rtb_start:
	and.w	#$fffe,d1
	lea	(a2,d1.w),a3
	ror.w	#4,d1
	move.w	d1,d0
	and.w	#$ff,d1
	move.w	d1,d3
	ror.w	#3,d0
	and.w	#7*$100*4,d0
	add.w	d1,d1
	add.w	d1,d1
	add.w	d1,d0		;prepare some values

rtb_check:
	move.w	bg_scx_pixel(a5),d2
	lea	bg_used,a2
	cmp.b	(a2,d3.w),d2
	bne.s	rtb_exit	;tile not used?

	lea	bg_tile,a2
	add.w	d0,a2

	move.l	a0,-(sp)
	moveq	#0,d0
	move.l	bw_color_l(a5),a0
	jsr	(a0)
	swap	d0
	move.l	bw_color_h(a5),a0
	jsr	(a0)
	lsl.l	d2,d0
	movep.w	d0,1(a2)
	swap	d0
	movep.w	d0,0(a2)
	moveq	#0,d1
	move.b	(a3)+,d1
	or.b	(a3),d1
	lsl.w	d2,d1
	move.w	d1,8*$100*4(a2)
	move.l	(sp)+,a0	;convert one line
rtb_exit:
	move.w	(sp)+,d3
rtb_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into tile table for background and window

ram_tile_bw:
	cmp.b	(a2,d1.w),d0
	beq.s	rtb_quit	;no change?

	move.w	d3,-(sp)

	move.b	d0,(a2,d1.w)	;store pattern in tile table
rtbw_start:
	and.w	#$fffe,d1
	lea	(a2,d1.w),a3
	ror.w	#4,d1
	move.w	d1,d0
	and.w	#$ff,d1
	move.w	d1,d3
	ror.w	#3,d0
	and.w	#7*$100*4,d0
	add.w	d1,d1
	add.w	d1,d1
	add.w	d1,d0		;prepare some values

	move.w	wi_wx_pixel(a5),d2
	lea	wi_used,a2
	cmp.b	(a2,d3.w),d2
	bne.s	rtbw_quit	;tile (wi) not used?

	lea	wi_tile,a2
	add.w	d0,a2

	move.l	a0,-(sp)
	move.w	d0,-(sp)
	moveq	#0,d0
	move.l	bw_color_l(a5),a0
	jsr	(a0)
	swap	d0
	move.l	bw_color_h(a5),a0
	jsr	(a0)
	lsl.l	d2,d0
	movep.w	d0,1(a2)
	swap	d0
	movep.w	d0,0(a2)
	moveq	#0,d1
	move.b	(a3),d1
	or.b	1(a3),d1
	lsl.w	d2,d1
	move.w	d1,8*$100*4(a2)
	move.w	(sp)+,d0
	move.l	(sp)+,a0	;convert one line (wi)
rtbw_quit:
	bra	rtb_check	;now check for background

;----------------------------------------------------------------
;	write into tile table for objects

ram_tile_ob:
	cmp.b	(a2,d1.w),d0
	beq.s	rto_quit	;no change?

	move.b	d0,(a2,d1.w)	;store pattern in tile table

	move.w	d3,-(sp)
	bsr	ob_tile_check
	move.w	(sp)+,d3
rto_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into tile table for background and objects

ram_tile_bo:
	cmp.b	(a2,d1.w),d0
	beq.s	rto_quit	;no change?

	move.b	d0,(a2,d1.w)	;store pattern in tile table

	move.w	d3,-(sp)
	bsr	ob_tile_check
	bra	rtb_start

;----------------------------------------------------------------
;	write into tile table for background, window, and objects

ram_tile_bwo:
	cmp.b	(a2,d1.w),d0
	beq.s	rto_quit	;no change?

	move.b	d0,(a2,d1.w)	;store pattern in tile table

	move.w	d3,-(sp)
	bsr	ob_tile_check
	bra	rtbw_start

;----------------------------------------------------------------
;	write into tile map without watching

ram_map_none:
	cmp.b	(a2,d1.w),d0
	beq.s	rmn_quit	;no change?

	move.b	d0,(a2,d1.w)	;store tile number in map

	add.w	d1,d1
	and.w	#2*MAP_WRAP*MAP_WRAP*2-1,d1
	lea	map_indices,a2
	moveq	#0,d2
	move.b	d0,d2
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,(a2,d1.w)	;store precalculated index
rmn_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into background tile map

ram_map_bg:
	cmp.b	(a2,d1.w),d0
	beq.s	rmb_quit	;no change?

	move.b	d0,(a2,d1.w)	;store tile number in map

	add.w	d1,d1
	and.w	#2*MAP_WRAP*MAP_WRAP*2-1,d1
	lea	map_indices,a2
	moveq	#0,d2
	move.b	d0,d2
	move.w	d2,d0
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,(a2,d1.w)	;store precalculated index

	move.w	d2,d1
	move.w	bg_scx_pixel(a5),d2
	lea	bg_used,a2
	cmp.b	(a2,d0.w),d2
	beq.s	rmb_quit	;already converted?
	move.b	d2,(a2,d0.w)

	move.l	bw_tile_low(a5),a3
	tst.b	d0
	bpl.s	rmb_tile_ok
	move.l	bw_tile_high(a5),a3
rmb_tile_ok:
	lea	bg_tile,a2
	add.w	d1,a2
	add.w	d1,d1
	add.w	d1,d1
	add.w	d1,a3
	movem.l	d3/a0-a1,-(sp)
	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	bsr	convert_bw
	movem.l	(sp)+,d3/a0-a1

rmb_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into window tile map

ram_map_wi:
	cmp.b	(a2,d1.w),d0
	beq.s	rmw_quit	;no change?

	move.b	d0,(a2,d1.w)	;store tile number in map

	add.w	d1,d1
	and.w	#2*MAP_WRAP*MAP_WRAP*2-1,d1
	lea	map_indices,a2
	moveq	#0,d2
	move.b	d0,d2
	move.w	d2,d0
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,(a2,d1.w)	;store precalculated index

	move.w	d2,d1
	move.w	wi_wx_pixel(a5),d2
	lea	wi_used,a2
	cmp.b	(a2,d0.w),d2
	beq.s	rmw_quit	;already converted?
	move.b	d2,(a2,d0.w)

	move.l	bw_tile_low(a5),a3
	tst.b	d0
	bpl.s	rmw_tile_ok
	move.l	bw_tile_high(a5),a3
rmw_tile_ok:
	lea	wi_tile,a2
	add.w	d1,a2
	add.w	d1,d1
	add.w	d1,d1
	add.w	d1,a3
	movem.l	d3/a0-a1,-(sp)
	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	bsr	convert_bw
	movem.l	(sp)+,d3/a0-a1

rmw_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into background and window tile map

ram_map_bw:
	cmp.b	(a2,d1.w),d0
	beq	rmbw_wi_quit	;no change?

	move.l	bw_tile_low(a5),a3
	move.b	d0,(a2,d1.w)	;store tile number in map
	bpl.s	rmbw_tile_ok
	move.l	bw_tile_high(a5),a3
rmbw_tile_ok:

	add.w	d1,d1
	and.w	#2*MAP_WRAP*MAP_WRAP*2-1,d1
	lea	map_indices,a2
	moveq	#0,d2
	move.b	d0,d2
	move.w	d2,d0
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,(a2,d1.w)	;store precalculated index

	move.w	d0,-(sp)
	move.w	d2,d1
	add.w	d2,d2
	add.w	d2,d2
	add.w	d2,a3		;prepare some values

	move.w	bg_scx_pixel(a5),d2
	lea	bg_used,a2
	cmp.b	(a2,d0.w),d2
	beq.s	rmbw_bg_quit	;already converted?
	move.b	d2,(a2,d0.w)

	lea	bg_tile,a2
	add.w	d1,a2
	movem.l	d3/a0-a1,-(sp)
	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	bsr	convert_bw	;convert for background
	lea	-8*2(a3),a3
	movem.l	(sp)+,d3/a0-a1
rmbw_bg_quit:
	move.w	(sp)+,d0

	move.w	wi_wx_pixel(a5),d2
	lea	wi_used,a2
	cmp.b	(a2,d0.w),d2
	beq.s	rmbw_wi_quit	;already converted?
	move.b	d2,(a2,d0.w)

	add.w	d0,d0
	add.w	d0,d0
	lea	wi_tile,a2
	add.w	d0,a2
	movem.l	d3/a0-a1,-(sp)
	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	bsr	convert_bw	;convert for window
	movem.l	(sp)+,d3/a0-a1
rmbw_wi_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write into object attribute memory (oam)

ram_oam:
	pea	(a6)		;save return address

	moveq	#$fc-$100,d2
	and.w	d1,d2
	lea	(a2,d2.w),a3
	move.l	(a3),d2
	move.b	d0,(a2,d1.w)	;store byte in memory
	move.l	(a3),d0

	cmp.b	#40<<2,d1
	bcc.s	roe_quit	;object # too big?

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	new entry in oam

ROE_REGS	reg	d3-d5/a0/a1
;d0.l	new value (already stored)
;d1.b	offset into oam
;d2.l	old value
;a2	(scratch)
;a3	pointer into oam
;a6	(scratch)

ram_oam_entry:
	eor.l	d0,d2
	beq.s	roe_quit	;no change?

	movem.l	#ROE_REGS,-(sp)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	check changes in y or x

	move.l	#$ffff7f,d5
	and.l	d2,d5		;save changes in x/tile/flags

	lea	ob_tab,a6
	move.w	#$00fc,d0
	and.w	d1,d0
	move.w	d0,d1
	add.w	d0,d0
	add.w	d1,d0
	lsl.w	#4,d0
	add.w	#SCR_HEIGHT*8*2*2,d0
	lea	(a6,d0.w),a2	;prepare some values

	eor.w	d5,d2
	bne.s	roe_xy_link
	swap	d2
	tst.w	d2
	beq.s	roe_xy_quit	;no change?
roe_xy_link:
	bsr	ob_link		;re-link object
roe_xy_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	check changes in x, tile #, or flags

	tst.l	d5
	beq.s	roe_xtf_quit	;no change?

	move.w	(a3)+,d4
	move.w	ob_tilemask(a5),d3
	and.b	(a3)+,d3
	move.b	(a3)+,d0
	lsl.w	#3+1,d3
	move.l	BASE_RAM_VID(a4),a3
	lea	$8000(a3),a3
	add.w	d3,a3
	move.w	ob_count(a5),d3
	moveq	#$10,d2
	and.b	d0,d2
	bne.s	roe_xtf_obp1
	move.l	ob_color0_l(a5),a0
	move.l	ob_color0_h(a5),a1
	bra.s	roe_xtf_conv
roe_xtf_obp1:
	move.l	ob_color1_l(a5),a0
	move.l	ob_color1_h(a5),a1
roe_xtf_conv:
	bsr	convert_ob
roe_xtf_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	movem.l	(sp)+,#ROE_REGS
roe_quit:
	rts

;================================================================
;	write to LCD control

IWL_REGS	reg	d3/a0-a1
;d0	new contents
;d1	changed contents
;d2	(scratch)
;d3	(saved, mostly counter)
;a0	(saved, scratch)
;a1	(saved, scratch)
;a2	(scratch)
;a3	(scratch)
;a6	(scratch)

io_wr_lcdc:
	pea	(a6)		;save return address

	move.b	(a3),d1
	eor.b	d0,d1
	beq	iwl_nop		;no changes?
	move.b	d0,(a3)

	movem.l	#IWL_REGS,-(sp)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 7

	moveq	#7,d2		;lcd on: 0/1 = off/on
	btst	d2,d0
	bne.s	iwl7_on		;lcd on?

	btst	d2,d1
	beq	iwl_quit	;lcd still off?

	lea	ram_none(pc),a3
	move.l	a3,V_RAM_WR8087+2(a4)
	move.l	a3,V_RAM_WR888F+2(a4)
	move.l	a3,V_RAM_WR9097+2(a4)
	move.l	a3,V_RAM_WR_OAM+2(a4)
	lea	ram_map_none(pc),a3
	move.l	a3,V_RAM_WR989B+2(a4)
	move.l	a3,V_RAM_WR9C9F+2(a4)

	lea	line_none(pc),a6
	move.l	a6,line_function(a5)

	moveq	#0,d2
	move.b	d2,int_mask_vsync
	move.b	d2,int_mask_stat
	bra	iwl_quit

iwl7_on:
	btst	d2,d1
	beq	iwl7_quit	;lcd still on?

	or.b	#$01,d1
	and.b	#$fb,d1		;force change bit 0 on / 2 off

	btst	#2,d0
	beq.s	iwl7_small
	move.w	#16-1,ob_count(a5)
	move.w	#$fe,ob_tilemask(a5)
	bra.s	iwl7_goon
iwl7_small:
	move.w	#8-1,ob_count(a5)
	move.w	#$ff,ob_tilemask(a5)	;set up some variables

iwl7_goon:
	movem.l	d0/d1/d4/d5,-(sp)
	bsr	ob_link_convert_all
	movem.l	(sp)+,d0/d1/d4/d5

	lea	ram_oam(pc),a3
	move.l	a3,V_RAM_WR_OAM+2(a4)

	move.w	#SCR_YMAX-153,LCD_I_LY(a4)
	lea	line_gap(pc),a6
	move.l	a6,V_LCD_LINE+2(a4)

	move.b	#01,int_mask_vsync
	move.b	#02,int_mask_stat
iwl7_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 0

	moveq	#0,d2		;bg/win on: 0/1 = off/on
	btst	d2,d0
	bne.s	iwl0_on		;background/window on?

	btst	d2,d1
	beq	iwl1_bw_off	;background/window still off?

	lea	ram_tile_ob(pc),a3
	move.l	a3,V_RAM_WR8087+2(a4)
	move.l	a3,V_RAM_WR888F+2(a4)
	lea	ram_none(pc),a3
	move.l	a3,V_RAM_WR9097+2(a4)
	lea	ram_map_none(pc),a3
	move.l	a3,V_RAM_WR989B+2(a4)
	move.l	a3,V_RAM_WR9C9F+2(a4)

	lea	line_none(pc),a6
	btst	#1,d0
	beq.s	iwl0_ob_off	;objects off?
	move.w	#SCR_YMAX,d2
	sub.w	LCD_I_LY(a4),d2
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,ob_y_offset(a5)
	lea	line_ob(pc),a6
iwl0_ob_off:
	move.l	a6,line_function(a5)
	bra	iwl1_bw_off

iwl0_on:
	btst	d2,d1
	beq.s	iwl0_quit	;background/window still on?

	or.b	#$38,d1		;force change bits 5..3

	move.w	#$100/4-1,d3
	lea	bg_used,a6
	moveq	#-1,d2
iwl0_loop:
	move.l	d2,(a6)+
	dbra	d3,iwl0_loop	;no background tiles used
iwl0_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 5

	moveq	#5,d2		;window on: 0/1 = off/on
	btst	d2,d0
	bne	iwl5_on		;window on?

	btst	d2,d1
	beq	iwl3_wi_off	;window still off?

	lea	ram_tile_ob(pc),a1
	lea	ram_tile_bo(pc),a2
	lea	ram_tile_bg(pc),a3
	btst	#4,d0
	beq.s	iwl5_signed	;tile type signed?
	move.l	a2,a1
	lea	ram_none(pc),a3
iwl5_signed:
	move.l	a1,V_RAM_WR8087+2(a4)
	move.l	a2,V_RAM_WR888F+2(a4)
	move.l	a3,V_RAM_WR9097+2(a4)

	lea	line_bg(pc),a6
	btst	#1,d0
	beq.s	iwl5_ob_off	;objects off?
	move.w	#SCR_YMAX,d2
	sub.w	LCD_I_LY(a4),d2
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,ob_y_offset(a5)
	lea	line_bo(pc),a6
iwl5_ob_off:
	move.l	a6,line_function(a5)

	lea	ram_map_bg(pc),a2
	lea	ram_map_none(pc),a6
	btst	#3,d0
	beq.s	iwl5_low	;background at lower address?
	exg	a2,a6
iwl5_low:
	move.l	a2,V_RAM_WR989B+2(a4)
	move.l	a6,V_RAM_WR9C9F+2(a4)
	bra	iwl3_wi_off

iwl5_on:
	btst	d2,d1
	beq	iwl5_quit	;window still on?

	or.b	#$40,d1		;force change bit 6

	moveq	#0,d2
	move.b	LCD_WY(a4),d2
	add.w	LCD_I_LY(a4),d2
	sub.w	#SCR_YMAX-1+1,d2
	bpl.s	iwl5_w_counter
	swap	d2
iwl5_w_counter:
	move.w	d2,LCD_I_WY(a4)	;lines without window

	lea	ram_tile_ob(pc),a1
	lea	ram_tile_bwo(pc),a2
	lea	ram_tile_bw(pc),a3
	btst	#4,d0
	beq.s	iwl5_w_signed	;tile type signed?
	move.l	a2,a1
	lea	ram_none(pc),a3
iwl5_w_signed:
	move.l	a1,V_RAM_WR8087+2(a4)
	move.l	a2,V_RAM_WR888F+2(a4)
	move.l	a3,V_RAM_WR9097+2(a4)

	lea	line_bc(pc),a6
	btst	#1,d0
	beq.s	iwl5_w_ob_off	;objects off?
	move.w	#SCR_YMAX,d2
	sub.w	LCD_I_LY(a4),d2
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,ob_y_offset(a5)
	lea	line_bco(pc),a6
iwl5_w_ob_off:
	move.l	a6,line_function(a5)

	move.w	#$100/4-1,d3
	lea	wi_used,a6
	moveq	#-1,d2
iwl5_loop:
	move.l	d2,(a6)+
	dbra	d3,iwl5_loop	;no window tiles used
iwl5_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 6

	moveq	#6,d2		;window base: 0/1 = $9800/$9c00
	btst	d2,d1
	beq	iwl6_quit	;window base not changed?

	move.l	BASE_RAM_VID(a4),a3
	lea	map_indices,a0

	btst	d2,d0
	beq.s	iwl6_low	;at lower address?

	move.w	#MAP_WRAP*MAP_WRAP,d2

	btst	#3,d0
	beq.s	iwl6_wh_bl	;background at lower address?

	lea	ram_map_none(pc),a2
	lea	ram_map_bw(pc),a6
	bra.s	iwl6_goon
iwl6_wh_bl:
	lea	ram_map_bg(pc),a2
	lea	ram_map_wi(pc),a6
	bra.s	iwl6_goon

iwl6_low:
	moveq	#0,d2

	btst	#3,d0
	beq.s	iwl6_wl_bl	;background at lower address?

	lea	ram_map_wi(pc),a2
	lea	ram_map_bg(pc),a6
	bra.s	iwl6_goon
iwl6_wl_bl:
	lea	ram_map_bw(pc),a2
	lea	ram_map_none(pc),a6

iwl6_goon:
	move.l	a2,V_RAM_WR989B+2(a4)
	move.l	a6,V_RAM_WR9C9F+2(a4)

	lea	$9800(a3),a3
	add.w	d2,a3

	add.w	d2,d2
	add.w	d2,a0
	move.l	a0,wi_map_start(a5)

	move.w	d0,-(sp)
	move.w	d1,-(sp)

	moveq	#0,d0
	moveq	#$7f,d1
	move.w	wi_wx_pixel(a5),d2
	move.w	#SCR_HEIGHT*MAP_WRAP-1,d3
	lea	wi_used,a6
iwl6_loop:
	move.b	(a3)+,d0
	cmp.b	(a6,d0.w),d2
	beq.s	iwl6_next
	move.b	d1,(a6,d0.w)
iwl6_next:
	dbra	d3,iwl6_loop	;mark which tiles need conversion

	move.w	(sp)+,d1
	move.w	(sp)+,d0
iwl6_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 3, two entries: window on or off

iwl3_wi_on:
	moveq	#3,d2		;bg. base: 0/1 = $9800/$9c00
	btst	d2,d1
	beq	iwl3_quit	;background base not changed?

	move.l	BASE_RAM_VID(a4),a3
	lea	map_indices,a0

	btst	d2,d0
	beq.s	iwl3_w_low	;at lower address?

	move.w	#MAP_WRAP*MAP_WRAP,d2

	btst	#6,d0
	beq.s	iwl3_wl_bh	;window at lower address?

	lea	ram_map_none(pc),a2
	lea	ram_map_bw(pc),a6
	bra.s	iwl3_scan
iwl3_wl_bh:
	lea	ram_map_wi(pc),a2
	lea	ram_map_bg(pc),a6
	bra.s	iwl3_scan

iwl3_w_low:
	moveq	#0,d2

	btst	#6,d0
	beq.s	iwl3_wl_bl	;window at lower address?

	lea	ram_map_bg(pc),a2
	lea	ram_map_wi(pc),a6
	bra.s	iwl3_scan
iwl3_wl_bl:
	lea	ram_map_bw(pc),a2
	lea	ram_map_none(pc),a6
	bra.s	iwl3_scan

iwl3_wi_off:
	moveq	#3,d2		;bg. base: 0/1 = $9800/$9c00
	btst	d2,d1
	beq	iwl3_quit	;background base not changed?

	move.l	BASE_RAM_VID(a4),a3
	lea	map_indices,a0

	btst	d2,d0
	beq.s	iwl3_bl		;at lower address?

	move.w	#MAP_WRAP*MAP_WRAP,d2

	lea	ram_map_none(pc),a2
	lea	ram_map_bg(pc),a6
	bra.s	iwl3_scan
iwl3_bl:
	moveq	#0,d2

	lea	ram_map_bg(pc),a2
	lea	ram_map_none(pc),a6

iwl3_scan:
	move.l	a2,V_RAM_WR989B+2(a4)
	move.l	a6,V_RAM_WR9C9F+2(a4)

	lea	$9800(a3),a3
	add.w	d2,a3

	add.w	d2,d2
	add.w	d2,a0
	move.l	a0,bg_map_start(a5)

	moveq	#SCR_YMAX-$100,d2
	sub.w	LCD_I_LY(a4),d2
	add.b	LCD_SCY(a4),d2
	add.w	d2,d2
	add.w	d2,d2
	move.b	d2,-(sp)
	and.w	#(MAP_WRAP-1)*MAP_WRAP,d2
	add.w	bg_scx_tile(a5),d2
	add.w	d2,d2
	move.w	d2,bg_map_offset(a5)
	move.w	(sp)+,d2
	and.w	#7*$100*4,d2
	lea	bg_tile,a2
	add.w	d2,a2
	move.l	a2,bg_tile_ptr(a5)	;correct actual pointer

	move.w	d0,-(sp)
	move.w	d1,-(sp)

	moveq	#0,d0
	moveq	#$7f,d1
	move.w	bg_scx_pixel(a5),d2
	move.w	#MAP_WRAP*MAP_WRAP-1,d3
	lea	bg_used,a6
iwl3_loop:
	move.b	(a3)+,d0
	cmp.b	(a6,d0.w),d2
	beq.s	iwl3_next
	move.b	d1,(a6,d0.w)
iwl3_next:
	dbra	d3,iwl3_loop	;mark which tiles need conversion

	move.w	(sp)+,d1
	move.w	(sp)+,d0
iwl3_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 4

	moveq	#4,d2		;tile type: 0/1 = signed/unsigned
	btst	d2,d1
	beq	iwl4_quit	;tile type not changed?

	btst	#5,d0
	bne.s	iwl4_wi_on	;window on?

	lea	ram_tile_ob(pc),a1
	lea	ram_tile_bo(pc),a2
	lea	ram_tile_bg(pc),a3
	bra.s	iwl4_goon

iwl4_wi_on:
	lea	ram_tile_ob(pc),a1
	lea	ram_tile_bwo(pc),a2
	lea	ram_tile_bw(pc),a3

	or.b	#$40,d1		;window tiles need conversion

	moveq	#$7f,d2
	moveq	#$80-1,d3
	lea	wi_used,a6
iwl4_wi_loop:
	tst.b	(a6)+
	bmi.s	iwl4_wi_next
	move.b	d2,-1(a6)
iwl4_wi_next:
	dbra	d3,iwl4_wi_loop	;mark which tiles need conversion

iwl4_goon:
	move.l	BASE_RAM_VID(a4),a6
	lea	$8000(a6),a6
	move.l	a6,bw_tile_high(a5)

	lea	$1000(a6),a6
	btst	#4,d0
	beq.s	iwl4_signed	;tile type signed?
	move.l	a2,a1
	lea	ram_none(pc),a3
	lea	-$1000(a6),a6
iwl4_signed:
	move.l	a1,V_RAM_WR8087+2(a4)
	move.l	a2,V_RAM_WR888F+2(a4)
	move.l	a3,V_RAM_WR9097+2(a4)
	move.l	a6,bw_tile_low(a5)

	moveq	#$7f,d2
	moveq	#$80-1,d3
	lea	bg_used,a6
iwl4_bg_loop:
	tst.b	(a6)+
	bmi.s	iwl4_bg_next
	move.b	d2,-1(a6)
iwl4_bg_next:
	dbra	d3,iwl4_bg_loop	;mark which tiles need conversion

	or.b	#$08,d1		;background tiles need conversion
iwl4_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert background tiles if necessary

	moveq	#3,d2		;background tiles need conversion
	btst	d2,d1
	beq.s	iwl_bg_quit	;no conversions?

	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	move.w	bg_scx_pixel(a5),d2
	lea	bg_used,a6
	lea	bg_tile,a2
	bsr	convert_bw_all_marked
iwl_bg_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	convert window tiles if necessary

	moveq	#6,d2		;window tiles need conversion
	btst	d2,d1
	beq.s	iwl_wi_quit	;no conversions?

	move.l	bw_color_l(a5),a0
	move.l	bw_color_h(a5),a1
	move.w	wi_wx_pixel(a5),d2
	lea	wi_used,a6
	lea	wi_tile,a2
	bsr	convert_bw_all_marked
iwl_wi_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 1, two entries: background/window on or off

iwl1_bw_on:
	moveq	#1,d2		;objects on: 0/1 = off/on
	btst	d2,d1
	beq.s	iwl1_quit	;state not changed?

	btst	#5,d0
	beq.s	iwl1_wi_off	;window off?
	lea	line_bc(pc),a2
	lea	line_bco(pc),a3
	bra.s	iwl1_goon
iwl1_wi_off:
	lea	line_bg(pc),a2
	lea	line_bo(pc),a3
	bra.s	iwl1_goon

iwl1_bw_off:
	moveq	#1,d2		;objects on: 0/1 = off/on
	btst	d2,d1
	beq.s	iwl1_quit	;state not changed?

	lea	line_none(pc),a2
	lea	line_ob(pc),a3
iwl1_goon:
	btst	#1,d0
	beq.s	iwl1_ob_off	;objects off?
	move.w	#SCR_YMAX,d2
	sub.w	LCD_I_LY(a4),d2
	add.w	d2,d2
	add.w	d2,d2
	move.w	d2,ob_y_offset(a5)
	exg	a2,a3
iwl1_ob_off:
	move.l	a2,line_function(a5)
iwl1_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
;	handle bit 2

	moveq	#2,d2		;object height: 0/1 = 8/16
	btst	d2,d1
	beq.s	iwl2_quit	;height not changed?

	btst	d2,d0
	beq.s	iwl2_small
	move.w	#16-1,ob_count(a5)
	move.w	#$fe,ob_tilemask(a5)
	bra.s	iwl2_goon
iwl2_small:
	move.w	#8-1,ob_count(a5)
	move.w	#$ff,ob_tilemask(a5)	;set up some variables

iwl2_goon:
	movem.l	d0/d1/d4/d5,-(sp)
	bsr	ob_link_convert_all
	movem.l	(sp)+,d0/d1/d4/d5
iwl2_quit:

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwl_quit:
	movem.l	(sp)+,#IWL_REGS

	move.l	line_function(a5),a6
	move.w	LCD_I_LY(a4),d2
	bmi.s	iwl_nop		;not in visible part?
	move.l	a6,V_LCD_LINE+2(a4)

iwl_nop:
	rts

;----------------------------------------------------------------
;	read from LCD status and interrupt mode
;
;	these were correct timings, but don't work really and
;	slow down the emulation:
;	cmp.w	#(OPS_PER_LINE*(108-19-41))/108,d6
;	cmp.w	#(OPS_PER_LINE*(108-19))/108,d6

io_rd_stat:
	and.b	#$f8,(a3)

	moveq	#SCR_YMAX-1-$100,d0
	sub.w	LCD_I_LY(a4),d0
	cmp.b	#SCR_YMAX+1,d0
	bcs.s	irs_visible
	or.b	#$01,(a3)
	bra.s	irs_lyc

irs_visible:
	cmp.w	#OPS_PER_LINE-10,d6
	bcs.s	irs_lyc
	or.b	#$02,(a3)
	cmp.w	#OPS_PER_LINE-5,d6
	bcc.s	irs_lyc
	or.b	#$01,(a3)

irs_lyc:
	cmp.b	LCD_LYC(a4),d0
	bne.s	irs_quit
	or.b	#$04,(a3)

irs_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write to LCD status and interrupt mode

io_wr_stat:
	move.b	d0,(a3)		;store value

	move.b	#$00,int_mask_sv
	btst	#4,d0
	beq.s	iws_vsync_ok
	move.b	#$02,int_mask_sv
iws_vsync_ok:			;set up for "vsync"

	move.b	#$00,int_mask_sh
	moveq	#$28,d1
	and.b	d0,d1
	beq.s	iws_hsync_ok
	move.b	#$02,int_mask_sh
iws_hsync_ok:			;set up for "hsync"

	btst	#6,d0
	beq.s	iws_stat_ok
	moveq	#SCR_YMAX-1-$100,d1
	sub.w	LCD_I_LY(a4),d1
	cmp.b	LCD_LYC(a4),d1
	bne.s	iws_stat_ok
	move.b	int_mask_stat,d1
	or.b	d1,INT_IF(a4)
	pea	(a6)
	jmp	INT_CHECK(a4)	;check for "LYC"
iws_stat_ok:

	jmp	(a6)

;----------------------------------------------------------------
;	write to scroll y

io_wr_scy:
	move.b	d0,(a3)		;store value

	sub.w	LCD_I_LY(a4),d0
	add.b	#SCR_YMAX,d0
	add.w	d0,d0
	add.w	d0,d0
	move.b	d0,-(sp)
	and.w	#(MAP_WRAP-1)*MAP_WRAP,d0
	add.w	bg_scx_tile(a5),d0
	add.w	d0,d0
	move.w	d0,bg_map_offset(a5)
	move.w	(sp)+,d0
	and.w	#7*$100*4,d0
	lea	bg_tile,a3
	add.w	d0,a3
	move.l	a3,bg_tile_ptr(a5)	;correct actual pointer
	jmp	(a6)

;----------------------------------------------------------------
;	write to scroll x

io_wr_scx:
	move.l	d3,-(sp)

	move.b	d0,(a3)		;store value

	move.w	bg_scx_pixel(a5),d1
	moveq	#7,d2
	and.b	d0,d2
	move.w	d2,bg_scx_pixel(a5)
	lsr.b	#3,d0
	ext.w	d0
	move.w	d0,d3
	sub.w	bg_scx_tile(a5),d3
	move.w	d0,bg_scx_tile(a5)
	add.w	d3,d3
	add.w	d3,bg_map_offset(a5)	;update variables

	moveq	#SCR_WIDTH-1,d3
	sub.w	#MAP_WRAP-SCR_WIDTH,d0
	bmi.s	iwsx_counter_bg
	sub.w	d0,d3
	exg	d0,d3
iwsx_counter_bg:
	move.w	d0,bg_count1(a5)
	move.w	d3,bg_count2(a5)	;set up counters (bg)

	move.w	d0,bw_count1(a5)
	move.w	bw_count2(a5),d0
	bmi.s	iwsx_counter_exit
	move.w	bw_count3(a5),d0
	bmi.s	iwsx_counter_bw
	sub.w	d0,d3
	bpl.s	iwsx_counter_bw
	move.w	d3,bw_count1(a5)
	moveq	#SCR_WIDTH-1,d3
	sub.w	d0,d3
iwsx_counter_bw:
	move.w	d3,bw_count2(a5)	;set up counters (bw)
iwsx_counter_exit:

	moveq	#$81-$100,d0
	move.b	LCD_LCDC(a4),d3
	and.b	d0,d3
	cmp.b	d0,d3
	bne	iwswx_quit	;background off?

	lea	bg_used,a3
	lea	bg_tile,a2

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwswx_shift:
	sub.w	d2,d1
	beq	iwswx_quit	;no new pixel shift?
	bcc.s	iwswx_right	;shift right?

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	neg.w	d1
	move.w	#$100-1,d3
iwswx_l_loop:
	move.b	(a3)+,d0
	bmi.s	iwswx_l_next
	cmp.b	d0,d2
	bne.s	iwswx_l_corr	;correction necessary?
iwswx_l_next:
	addq.w	#4,a2
	dbra	d3,iwswx_l_loop
	bra	iwswx_quit
iwswx_l_corr:
	move.b	d2,-1(a3)

	swap	d3
	move.w	#8-1,d3
iwswx_l_iloop:
	movep.w	0(a2),d0
	swap	d0
	movep.w	1(a2),d0
	lsl.l	d1,d0
	movep.w	d0,1(a2)
	swap	d0
	movep.w	d0,0(a2)
	move.w	8*$100*4(a2),d0
	lsl.w	d1,d0
	move.w	d0,8*$100*4(a2)

	lea	$100*4(a2),a2

	dbra	d3,iwswx_l_iloop
	swap	d3
	lea	4-8*$100*4(a2),a2

	dbra	d3,iwswx_l_loop
	bra.s	iwswx_quit

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwswx_right:
	move.w	#$100-1,d3
iwswx_r_loop:
	move.b	(a3)+,d0
	bmi.s	iwswx_r_next
	cmp.b	d0,d2
	bne.s	iwswx_r_corr	;correction necessary?
iwswx_r_next:
	addq.w	#4,a2
	dbra	d3,iwswx_r_loop
	bra.s	iwswx_quit
iwswx_r_corr:
	move.b	d2,-1(a3)

	swap	d3
	move.w	#8-1,d3
iwswx_r_iloop:
	movep.w	0(a2),d0
	swap	d0
	movep.w	1(a2),d0
	lsr.l	d1,d0
	movep.w	d0,1(a2)
	swap	d0
	movep.w	d0,0(a2)
	move.w	8*$100*4(a2),d0
	lsr.w	d1,d0
	move.w	d0,8*$100*4(a2)

	lea	$100*4(a2),a2

	dbra	d3,iwswx_r_iloop
	swap	d3
	lea	4-8*$100*4(a2),a2

	dbra	d3,iwswx_r_loop

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwswx_quit:
	move.l	(sp)+,d3
	jmp	(a6)

;----------------------------------------------------------------
;	write to LCD y compare

io_wr_lyc:
	move.b	d0,(a3)		;store value

	move.b	d0,int_lyc_reload
	moveq	#SCR_YMAX-$100,d1
	sub.w	LCD_I_LY(a4),d1
	sub.b	d1,d0
	move.b	d0,int_lyc_count	;set up counter

	addq.b	#1,d0
	bne.s	iwlc_quit
	btst	#6,LCD_STAT(a4)
	beq.s	iwlc_quit
	move.b	int_mask_stat,d0
	or.b	d0,INT_IF(a4)
	pea	(a6)
	jmp	INT_CHECK(a4)	;check for "LYC"

iwlc_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write to dma register

IWD_REGS	reg	d3/a0-a1

io_wr_dma:
	movem.l	#IWD_REGS,-(sp)

	move.l	BASE_RAM_OAM(a4),a1
	lea	$fe00+40*4(a1),a1
	lea	40*4(a3),a0
	moveq	#40*4-4-$100,d3
iwd_loop:
	move.l	-(a1),d2
	move.l	-(a0),d0
	move.l	d0,(a1)
	move.b	d3,d1
	move.l	a1,a3
	bsr	ram_oam_entry
	subq.b	#4,d3
	bcc.s	iwd_loop

	movem.l	(sp)+,#IWD_REGS
	rts

;----------------------------------------------------------------
;	write to background/window palette

IWB_REGS	reg	d3/a0-a1

io_wr_bgp:
	pea	(a6)		;save return address

	cmp.b	(a3),d0
	beq	iwb_quit	;no changes?
	move.b	d0,(a3)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	moveq	#0,d1
	moveq	#0,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2		;calculate color indices

	lea	color_tab(pc),a3
	add.w	d1,d1
	add.w	(a3,d1.w),d1
	lea	(a3,d1.w),a2
	move.l	a2,bw_color_h(a5)
	add.w	d2,d2
	add.w	(a3,d2.w),d2
	add.w	d2,a3
	move.l	a3,bw_color_l(a5)	;set up color functions

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	moveq	#$81-$100,d0
	move.b	LCD_LCDC(a4),d1
	and.b	d0,d1
	cmp.b	d0,d1
	bne	iwb_quit	;background/window off?

	movem.l	#IWB_REGS,-(sp)

	move.l	a3,a0
	move.l	a2,a1

	lea	bg_used,a6
	lea	bg_tile,a2
	move.w	bg_scx_pixel(a5),d2
	bsr	convert_bw_all_used

	btst	#5,LCD_LCDC(a4)
	beq.s	iwb_exit	;window off?

	lea	wi_used,a6
	lea	wi_tile,a2
	move.w	wi_wx_pixel(a5),d2
	bsr	convert_bw_all_used

iwb_exit:
	movem.l	(sp)+,#IWB_REGS

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwb_quit:
	rts

;----------------------------------------------------------------
;	write to object palette 0

IWO_REGS	reg	d3-d5/a0-a1

io_wr_obp0:
	pea	(a6)		;save return address

	cmp.b	(a3),d0
	beq	iwo0_quit	;no changes?
	move.b	d0,(a3)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	moveq	#0,d1
	moveq	#0,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2		;calculate color indices

	lea	color_tab(pc),a3
	add.w	d1,d1
	add.w	(a3,d1.w),d1
	lea	(a3,d1.w),a2
	move.l	a2,ob_color0_h(a5)
	add.w	d2,d2
	add.w	(a3,d2.w),d2
	add.w	d2,a3
	move.l	a3,ob_color0_l(a5)	;set up color functions

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	btst	#7,LCD_LCDC(a4)
	beq.s	iwo0_quit	;lcd off?

	movem.l	#IWO_REGS,-(sp)

	move.l	a3,a0
	move.l	a2,a1
	lea	ob_tab+SCR_HEIGHT*8*2*2,a2
	move.l	BASE_RAM_VID(a4),a3
	lea	$8000(a3),a3
	move.l	BASE_RAM_OAM(a4),a6
	lea	$fe00(a6),a6

	moveq	#40-1,d3
iwo0_loop:
	move.w	(a6)+,d4
	move.w	ob_tilemask(a5),d1
	and.b	(a6)+,d1
	move.b	(a6)+,d0
	moveq	#$10,d2
	and.b	d0,d2
	bne.s	iwo0_next	;not this palette?

	swap	d3
	movem.l	a2/a3/a6,-(sp)
	lsl.w	#3+1,d1
	add.w	d1,a3
	move.w	ob_count(a5),d3
	bsr	convert_ob
	movem.l	(sp)+,a2/a3/a6
	swap	d3
iwo0_next:
	lea	16*6*2(a2),a2
	dbra	d3,iwo0_loop	;convert object with new palette

iwo0_exit:
	movem.l	(sp)+,#IWO_REGS

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwo0_quit:
	rts

;----------------------------------------------------------------
;	write to object palette 1

io_wr_obp1:
	pea	(a6)		;save return address

	cmp.b	(a3),d0
	beq	iwo1_quit	;no changes?
	move.b	d0,(a3)

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	moveq	#0,d1
	moveq	#0,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2
	add.b	d0,d0
	addx.b	d1,d1
	add.b	d0,d0
	addx.b	d2,d2		;calculate color indices

	lea	color_tab(pc),a3
	add.w	d1,d1
	add.w	(a3,d1.w),d1
	lea	(a3,d1.w),a2
	move.l	a2,ob_color1_h(a5)
	add.w	d2,d2
	add.w	(a3,d2.w),d2
	add.w	d2,a3
	move.l	a3,ob_color1_l(a5)	;set up color functions

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

	btst	#7,LCD_LCDC(a4)
	beq.s	iwo1_quit	;lcd off?

	movem.l	#IWO_REGS,-(sp)

	move.l	a3,a0
	move.l	a2,a1
	lea	ob_tab+SCR_HEIGHT*8*2*2,a2
	move.l	BASE_RAM_VID(a4),a3
	lea	$8000(a3),a3
	move.l	BASE_RAM_OAM(a4),a6
	lea	$fe00(a6),a6

	moveq	#40-1,d3
iwo1_loop:
	move.w	(a6)+,d4
	move.w	ob_tilemask(a5),d1
	and.b	(a6)+,d1
	move.b	(a6)+,d0
	moveq	#$10,d2
	and.b	d0,d2
	beq.s	iwo1_next	;not this palette?

	swap	d3
	movem.l	a2/a3/a6,-(sp)
	lsl.w	#3+1,d1
	add.w	d1,a3
	move.w	ob_count(a5),d3
	bsr	convert_ob
	movem.l	(sp)+,a2/a3/a6
	swap	d3
iwo1_next:
	lea	16*6*2(a2),a2
	dbra	d3,iwo1_loop	;convert object with new palette

iwo1_exit:
	movem.l	(sp)+,#IWO_REGS

; - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

iwo1_quit:
	rts

;----------------------------------------------------------------
;	write to window y coordinate

io_wr_wy:
	move.b	d0,(a3)		;store value

	moveq	#$a1-$100,d1
	move.b	LCD_LCDC(a4),d2
	and.b	d1,d2
	cmp.b	d1,d2
	bne.s	iwwy_quit	;window off?

	move.w	LCD_I_LY(a4),d2
	bmi.s	iwwy_quit	;not in visible part?

	lea	line_bc(pc),a3
	btst	#1,LCD_LCDC(a4)
	beq.s	iwwy_ob_off	;objects off?
	lea	line_bco(pc),a3
iwwy_ob_off:
	move.l	a3,V_LCD_LINE+2(a4)

	and.w	#$ff,d0
	add.w	d2,d0
	sub.w	#SCR_YMAX-1+1,d0
	move.w	d0,LCD_I_WY(a4)
	bmi.s	iwwy_quit	;window shown yet?

	move.w	#0,wi_map_offset(a5)
	lea	wi_tile,a3
	move.l	a3,wi_tile_ptr(a5)	;correct actual pointer

iwwy_quit:
	jmp	(a6)

;----------------------------------------------------------------
;	write to window x coordinate

io_wr_wx:
	move.l	d3,-(sp)

	move.b	d0,(a3)		;store value

	move.w	wi_wx_pixel(a5),d1
	move.b	d0,d2
	not.b	d2
	and.w	#7,d2
	move.w	d2,wi_wx_pixel(a5)
	moveq	#-1,d3
	lsl.b	d2,d3
	move.b	d3,wi_wx_mask(a5)	;update variables

	moveq	#SCR_WIDTH-1,d3
	subq.b	#8,d0
	bcs.s	iwwx_full
	move.w	bg_count2(a5),d2
	cmp.b	#SCR_WIDTH*8-1,d0
	bcc.s	iwwx_empty
	lsr.b	#3,d0
	ext.w	d0
	sub.w	d0,d3
	move.w	bg_count1(a5),d0
	sub.w	d3,d2
	bpl.s	iwwx_counters
	moveq	#-1,d0
	moveq	#SCR_WIDTH-1,d2
	sub.w	d3,d2
	bra.s	iwwx_counters
iwwx_full:
	moveq	#-1,d2
	bra.s	iwwx_counters
iwwx_empty:
	move.w	bg_count1(a5),d0
	moveq	#-1,d3
iwwx_counters:
	move.w	d0,bw_count1(a5)
	move.w	d2,bw_count2(a5)
	move.w	d3,bw_count3(a5)	;set up counters

	moveq	#$a1-$100,d0
	move.b	LCD_LCDC(a4),d3
	and.b	d0,d3
	cmp.b	d0,d3
	bne	iwswx_quit	;window off?

	move.w	wi_wx_pixel(a5),d2
	lea	wi_used,a3
	lea	wi_tile,a2
	bra	iwswx_shift	;check for tile shifting

;================================================================
;	memory for screen emulation

		data
scr_imgs2:	dc.l	scr_img21,scr_img22,scr_img22,0
		bss
scr_img21:	ds.b	SCR_WIDTH*SCR_HEIGHT*8	;double buffering
scr_img22:	ds.b	SCR_WIDTH*SCR_HEIGHT*8

	;both of pattern and mask are
	;  8 pixel lines in
	;    $100 possible tiles of
	;      4 shifted and interlaced bytes
bg_used:	ds.b	$100		;which tiles are used
bg_tile:	ds.b	2*8*$100*4	;converted tiles
wi_used:	ds.b	$100		;which tiles are used
wi_tile:	ds.b	2*8*$100*4	;converted tiles

map_indices:	ds.w	2*MAP_WRAP*MAP_WRAP

	;(SCR_HEIGHT*8) lines with
	;  1 word offset to first background object
	;  1 word offset to first foreground object
	;40 objects of
	;  16 pixel lines with
	;    1 word screen horizontal offset
	;    1 word mask
	;    2 sets of
	;      2 bytes pattern
	;    1 word next offset
	;    1 word prev offset
		ds.w	1		;for offset -2
ob_tab:		ds.w	SCR_HEIGHT*8*2	;offsets to first object
		ds.w	40*16*6		;converted tiles

		offset
line_function:	ds.l	1	;address of next render function
bg_count1:	ds.w	1	;counter for loops, sum is 19
bg_count2:	ds.w	1
bw_count1:	ds.w	1	;if <0: no wrap before window
bw_count2:	ds.w	1	;if <0: only window
bw_count3:	ds.w	1	;if <0: window not visible

pic1:		ds.l	1	;pointer to lsb picture
pic2:		ds.l	1	;pointer to msb picture

bw_color_h:	ds.l	1	;addresses of color functions
bw_color_l:	ds.l	1
bw_tile_low:	ds.l	1	;pointer for tiles $00..$7f
bw_tile_high:	ds.l	1	;pointer for tiles $80..$ff
bw_mask:	ds.b	SCR_WIDTH	;mask buffer one line

bg_scx_pixel:	ds.w	1	;prehandled scroll
bg_scx_tile:	ds.w	1
bg_map_start:	ds.l	1	;pointer into background
bg_map_offset:	ds.w	1
bg_tile_ptr:	ds.l	1	;pointer to converted tiles
bg_tile_max:	ds.l	1

wi_wx_pixel:	ds.w	1	;prehandled scroll
wi_wx_mask:	ds.b	1
		even
wi_map_start:	ds.l	1	;pointer into window
wi_map_offset:	ds.w	1
wi_tile_ptr:	ds.l	1	;pointer to converted tiles
wi_tile_max:	ds.l	1

ob_y_offset:	ds.w	1	;offset into offset table
ob_color0_h:	ds.l	1	;addresses of color functions
ob_color0_l:	ds.l	1
ob_color1_h:	ds.l	1
ob_color1_l:	ds.l	1
ob_count:	ds.w	1	;prehandled height
ob_tilemask:	ds.w	1

sup_gl_size:	;label for size calculation
		bss
sup_globals:	ds.b	sup_gl_size

;================================================================
;	memory for interrupts

int_mask_vsync:	ds.b	1	;mask for vsync int
int_mask_stat:	ds.b	1	;mask for STAT int
int_mask_sv:	ds.b	1	;mask for STAT, vsync
int_mask_sh:	ds.b	1	;mask for STAT, hsync
int_lyc_reload:	ds.b	1	;counter for STAT, LYC
int_lyc_count:	ds.b	1
		even

;================================================================
;	memory for mbc emulation

mbc1_rom_lsb:	ds.w	1	;LSB of MBC1 rom select
mbc1_rom_msb:	ds.w	1	;MSB of MBC1 rom select

		offset
mbc3_rtc_run_v:	ds.b	1	;vsync-divider (must be first)
mbc3_rtc_run_s:	ds.b	1	;seconds
mbc3_rtc_run_m:	ds.b	1	;minutes
mbc3_rtc_run_h:	ds.b	1	;hours
mbc3_rtc_run_d:	ds.b	1	;days
mbc3_rtc_run_c:	ds.b	1	;control
mbc3_rtc_lat:	ds.b	5	;latched values

mbc3_rtc_size:	;label for size calculation
		bss
mbc3_rtc:	ds.b	mbc3_rtc_size
		even
mbc3_rtc_ptr:	ds.l	1	;pointer to current register

;================================================================
	end
