/****************************************************************
 *  win_cpu.c
 *			Window handling for CPU messages
 *
 *  2000-03-07  Bodo Wenzel  Creation in test
 *  2000-04-24  Bodo Wenzel  Copied to full version
 *  2000-06-06  Bodo Wenzel  New memory management
 *  2000-10-08  Bodo Wenzel  Interrupts
 *  2001-01-19  Bodo Wenzel  Support for big screens
 *  2001-02-10  Bodo Wenzel  Debug enhancements
 ****************************************************************

  (c)2000 Bodo Wenzel

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 ****************************************************************/

#define  noDEBUG  /* undefine for no additional line */		\
           " -- IE=%02X I_LY=%3d LYC=%3d STAT=%02X ----------",	\
           emulation[INT_VARIABLES].ext.int_vars.ie,		\
           143-emulation[LCD].ext.lcd_regs.i_ly,		\
           emulation[LCD].ext.lcd_regs.lyc,			\
           emulation[LCD].ext.lcd_regs.stat

/* === Includes ===============================================	*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aes.h>
#include <vdi.h>

#include "win_cpu.h"
#include "stemboy.h"
#include "emulate.h"
#include "gem_add.h"
#include "emu_wrap.h"

/* === Constants ==============================================	*/

#define  FALSE  (0)
#define  TRUE   (!FALSE)

#define  KIND  (NAME|CLOSER|FULLER|MOVER|INFO|SIZER| \
                UPARROW|DNARROW|VSLIDE|LFARROW|RTARROW|HSLIDE)

/* === Prototypes =============================================	*/

static int add_line(int handle,char *line);
static void set_slider(int handle);
static void *free_malloc(size_t n);

/* === Variables ==============================================	*/

typedef struct _m {
  struct _m  *n;
  struct _m  *p;
  char       *t;
} MSG;

static  MSG   *m_first=NULL,*m_last,*m_actual;
static  int   offs_x,offs_y,max_x,max_y,vis_x,vis_y;
static  char  *info;
static  int   wind_x,wind_y,wind_w=-1,wind_h;

/* === Functions ==============================================	*/

int cpu_open(void) {
  /*--------------*/
  int   x,y,w,h,handle;
  char  *s;

  wind_get(0,WF_WORKXYWH,&x,&y,&w,&h);
  if (wind_w<0) {
    wind_calc(WC_WORK,KIND,x,y+h/2,w,h/2,
              &wind_x,&wind_y,&wind_w,&wind_h);
    wind_x=((wind_x+wchr/2)/wchr)*wchr;
    wind_w=((wind_w+wchr/2)/wchr)*wchr;
    wind_h=((wind_h+hchr/2)/hchr)*hchr;
  }
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  x=((x+wchr/2)/wchr)*wchr;
  w=((w+wchr/2)/wchr)*wchr;
  h=((h+hchr/2)/hchr)*hchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  if ((handle=wind_create(KIND,x,y,w,h))<0)
    return -3;			/* can't create window */

  rsrc_gaddr(R_STRING,SCPUTITL,&s);
  wind_set(handle,WF_NAME,s);
  rsrc_gaddr(R_STRING,SCPUINFO,&info);
  wind_set(handle,WF_INFO,info);
  wind_calc(WC_BORDER,KIND,wind_x,wind_y,wind_w,wind_h,
            &x,&y,&w,&h);
  graf_mouse(M_OFF,0L);
  graf_growbox(0,0,0,0,x,y,w,h);
  wind_open(handle,x,y,w,h);
  graf_mouse(M_ON,0L);
  set_slider(handle);
  return handle;
}

void cpu_close(int handle) {
  /*----------------------*/
  int  x,y,w,h;

  wind_get(handle,WF_WORKXYWH,&wind_x,&wind_y,&wind_w,&wind_h);
  wind_get(handle,WF_CURRXYWH,&x,&y,&w,&h);
  graf_shrinkbox(0,0,0,0,x,y,w,h);
  wind_close(handle);
  wind_delete(handle);
}

void cpu_redraw(int handle,int rx,int ry,int rw,int rh) {
  /*---------------------------------------------------*/
  int   x,y,w,h,wx,wy,ww,wh,yy;
  int   pxy[8];
  MSG   *m;

  if (offs_x>=strlen(info))
    wind_set(handle,WF_INFO,"");
  else
    wind_set(handle,WF_INFO,info+offs_x);

  graf_mouse(M_OFF,0L);
  if (rw<0)
    wind_get(handle,WF_WORKXYWH,&rx,&ry,&rw,&rh);
  wind_get(handle,WF_WORKXYWH,&wx,&wy,&ww,&wh);
  wind_get(handle,WF_FIRSTXYWH,&x,&y,&w,&h);
  while (w!=0 && h!=0) {
    if (rc_intersect(rx,ry,rw,rh,&x,&y,&w,&h)) {
      pxy[0]=x;
      pxy[1]=y;
      pxy[2]=x+w-1;
      pxy[3]=y+h-1;
      vs_clip(vdi_handle,1,pxy);

      m=m_actual;
      for (yy=0; yy<wh; yy+=hchr) {
        pxy[2]=0;
        pxy[3]=0;
        if (m!=NULL) {
          if (m->t!=NULL)
            if (strlen(m->t)>offs_x) {
              v_gtext(vdi_handle,wx,wy+yy,m->t+offs_x);
              vqt_extent(vdi_handle,m->t+offs_x,pxy);
            }
          m=m->n;
        }
        pxy[0]=wx+pxy[2];
        pxy[1]=wy+yy+pxy[3];
        pxy[2]=wx+ww-1;
        pxy[3]=pxy[1]+hchr-1;
        if (pxy[0]<=pxy[2])
          v_bar(vdi_handle,pxy);
      }
    }
    wind_get(handle,WF_NEXTXYWH,&x,&y,&w,&h);
  }
  graf_mouse(M_ON,0L);
}

void cpu_sized(int handle,int x,int y,int w,int h) {
  /*----------------------------------------------*/
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  w=((w+wchr/2)/wchr)*wchr;
  h=((h+hchr/2)/hchr)*hchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  wind_set(handle,WF_CURRXYWH,x,y,w,h);
  set_slider(handle);
  cpu_redraw(handle,0,0,-1,0);
}

void cpu_fulled(int handle) {
  /*-----------------------*/
  int  cx,cy,cw,ch,x,y,w,h;

  wind_get(handle,WF_CURRXYWH,&cx,&cy,&cw,&ch);
  wind_get(handle,WF_FULLXYWH,&x,&y,&w,&h);
  if (cx==x && cy==y && cw==w && ch==h)
    wind_get(handle,WF_PREVXYWH,&x,&y,&w,&h);
  cpu_sized(handle,x,y,w,h);
}

void cpu_moved(int handle,int x,int y,int w,int h) {
  /*----------------------------------------------*/
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  x=((x+wchr/2)/wchr)*wchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  wind_set(handle,WF_CURRXYWH,x,y,w,h);
}

void cpu_vslided(int handle,int pos) {
  /*--------------------------------*/
  int  i;

  offs_y=(int)((long)pos*(max_y-vis_y)/1000);
  if (offs_y<0)
    offs_y=0;
  for (i=offs_y, m_actual=m_first; i-- && m_actual!=NULL;
       m_actual=m_actual->n);
  set_slider(handle);
  cpu_redraw(handle,0,0,-1,0);
}

void cpu_hslided(int handle,int pos) {
  /*--------------------------------*/
  offs_x=(int)((long)pos*(max_x-vis_x)/1000);
  if (offs_x<0)
    offs_x=0;
  set_slider(handle);
  cpu_redraw(handle,0,0,-1,0);
}

void cpu_arrowed(int handle,int arrow) {
  /*----------------------------------*/
  int  i;

  i=1;
  switch (arrow) {
  case WA_UPPAGE:
    i=vis_y;
  case WA_UPLINE:
    while (i--) {
      if (offs_y==0)
        break;
      m_actual=m_actual->p;
      offs_y--;
    }
    break;
  case WA_DNPAGE:
    i=vis_y;
  case WA_DNLINE:
    while (i--) {
      if (offs_y>=max_y-vis_y)
        break;
      m_actual=m_actual->n;
      offs_y++;
    }
    break;
  case WA_LFPAGE:
    i=vis_x;
  case WA_LFLINE:
    while (i--) {
      if (offs_x==0)
        break;
      offs_x--;
    }
    break;
  case WA_RTPAGE:
    i=vis_x;
  case WA_RTLINE:
    while (i--) {
      if (offs_x>=max_x-vis_x)
        break;
      offs_x++;
    }
    break;
  default:
    return;
  }
  set_slider(handle);
  cpu_redraw(handle,0,0,-1,0);
}

int cpu_add_line(int handle,EMU_COMBO *emulation,
                 EMU_STATE *state,CPU_REGS *regs) {
  /*---------------------------------------------*/
  int            oi;
  OBJECT         *o;
  OBSPEC         *f;
  char           r[100],l[500];
  unsigned char  *pc,*sp;

  switch ((int)state->run_result) {
  case ERR_DEBUG:     oi=DCCHECK;   break;
  case SINGLE_STEP:   oi=DCSSTEP;   break;
  case USER_STOP:     oi=DCUSTOP;   break;
  case OPCODE_STOP:   oi=DCCSTOP;   break;
  case OP_NOT_IMPL:   oi=DCOPNI;    break;
  case UNKNOWN:       oi=DCOPUNKN;  break;
  case NO_MEM:        oi=DCNOMEM;   break;
  case NO_INP:        oi=DCNOIN;    break;
  case NO_OUT:        oi=DCNOOUT;   break;
  case FTR_NOT_IMPL:  oi=DCFTRNI;   break;
  default:         if (state->run_result>=0 &&
                       state->run_result<=9)
                     oi=DCBREAKP;
                   else
                     oi=DCUNKN;
                   break;
  }
  rsrc_gaddr(R_TREE,DCPUMSG,&o);
  rsrc_gaddr(R_FRSTR,SCPUFORM,&f);
  sprintf(r,o[oi].ob_spec.free_string,state->run_result);

  pc=regs->pc_base+regs->pc;
  sp=regs->sp_base+regs->sp;
  sprintf(l,f->free_string,
    regs->pc,pc[-3],pc[-2],pc[-1],pc[0],pc[+1],pc[+2],
    regs->sp,sp[-2],sp[-1],sp[0],sp[+1],sp[+2],sp[+3],
    regs->a,
    state->int_enabled ? '*' : '-',
    (regs->f&FLAG_C) ? '*' : '-',(regs->f&FLAG_Z) ? '*' : '-',
    regs->b,regs->c,regs->d,regs->e,regs->h,regs->l,
    r);

#ifdef DEBUG
  if (!add_line(handle,l))
    return FALSE;
  
  sprintf(l,DEBUG);
#else
  (void)emulation;
#endif

  return add_line(handle,l);
}

static int add_line(int handle,char *line) {
  /*--------------------------------------*/
  MSG            *m;
  int            x,y;

  m=(MSG *)free_malloc(sizeof(MSG));
  if (m==NULL)
    return FALSE;

  m->n=NULL;
  m->p=m_last;

  m->t=(char *)free_malloc(strlen(line)+1);
  if (m->t==NULL) {
    free(m);
    return FALSE;
  }
  strcpy(m->t,line);
  if (m_last==NULL)
    m_first=m;
  else
    m_last->n=m;
  m_last=m;

  max_y++;
  offs_y=max_y-vis_y;
  if (offs_y<0)
    offs_y=0;
  for (m_actual=m, y=vis_y-1; y>0 && m_actual->p!=NULL; y--)
    m_actual=m_actual->p;
  x=(int)strlen(m->t);
  if (max_x<x)
    max_x=x;
  if (handle>=0) {
    set_slider(handle);
    cpu_redraw(handle,0,0,-1,0);
  }
  return TRUE;
}

void cpu_free_lines(int handle) {
  /*---------------------------*/
  MSG  *m1,*m2;

  graf_mouse(BUSYBEE,NULL);

  for (m1=m_first; m1!=NULL; m1=m2) {
    m2=m1->n;
    free(m1->t);
    free(m1);
  }

  m_first=NULL;
  m_last=NULL;
  m_actual=NULL;
  offs_x=0;
  offs_y=0;
  max_x=0;
  max_y=0;
  if (handle>=0) {
    set_slider(handle);
    cpu_redraw(handle,0,0,-1,0);
  }

  graf_mouse(ARROW,NULL);
}

static void set_slider(int handle) {
  /*------------------------------*/
  int  p,s,i,dummy;

  wind_get(handle,WF_WORKXYWH,&dummy,&dummy,&vis_x,&vis_y);

  vis_y/=hchr;
  if (max_y<=vis_y) {
    offs_y=0;
    m_actual=m_first;
    p=0;
    s=1000;
  } else {
    if (offs_y>max_y-vis_y) {
      offs_y=max_y-vis_y;
      for (i=offs_y, m_actual=m_first; i--; m_actual=m_actual->n);
    }
    p=(int)(1000L*offs_y/(max_y-vis_y));
    s=(int)(1000L*vis_y/max_y);
  }
  wind_set(handle,WF_VSLIDE,p);
  wind_set(handle,WF_VSLSIZE,s);

  vis_x/=wchr;
  if (max_x<=vis_x) {
    offs_x=0;
    p=0;
    s=1000;
  } else {
    if (offs_x>max_x-vis_x)
      offs_x=max_x-vis_x;
    p=(int)(1000L*offs_x/(max_x-vis_x));
    s=(int)(1000L*vis_x/max_x);
  }
  wind_set(handle,WF_HSLIDE,p);
  wind_set(handle,WF_HSLSIZE,s);
}

static void *free_malloc(size_t n) {
  /*------------------------------*/
  void  *p;

  for (;;) {
    p=malloc(n);
    if (p!=NULL)
      return p;			/* OK, got memory */

    if (max_y==0)
      return NULL;		/* no lines left to free */

    p=m_first->n;
    if (p!=NULL)
      ((MSG*)p)->p=NULL;
    max_y--;
    if (offs_y==0)
      m_actual=p;
    else
      offs_y--;
    free(m_first->t);
    free(m_first);
    m_first=p;			/* free first MSG */
  }
}

/* === End ====================================================	*/
