/****************************************************************
 *  win_prof.c
 *			Window handling for profiling output
 *
 *  2000-03-07  Bodo Wenzel  Creation in test
 *  2000-04-28  Bodo Wenzel  Copied to full version
 *  2000-08-22  Bodo Wenzel  Added evaluation of profile and
 *                           timing
 *  2001-01-19  Bodo Wenzel  Support for big screens
 ****************************************************************

  (c)2000 Bodo Wenzel

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 ****************************************************************/

/* === Includes ===============================================	*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aes.h>
#include <vdi.h>

#include "win_prof.h"
#include "stemboy.h"
#include "emulate.h"
#include "gem_add.h"
#include "emu_wrap.h"

/* === Constants ==============================================	*/

#define  DIV      (8)
#define  FORMAT   "%6.1f"
#define  F_EMPTY  "    . "
#define  F_SIZE   (6)
#define  ROWS     (0x100/DIV)
#define  COLS     (2+DIV*F_SIZE)

#if DIV!=8
#error "Look through the code, DIV isn't appropriate!"
#endif

#define  ACC_ROWS  (19)

#define  ALL_ROWS  (ROWS+ACC_ROWS)

#define  KIND  (NAME|CLOSER|FULLER|MOVER|INFO|SIZER| \
                UPARROW|DNARROW|VSLIDE|LFARROW|RTARROW|HSLIDE)

/* === Prototypes =============================================	*/

static void set_slider(int handle);
static void redraw(int handle,EMU_COMBO *emulation);

/* === Variables ==============================================	*/

static  char info[COLS+1]="  "
  " *0/*8 *1/*9 *2/*A *3/*B *4/*C *5/*D *6/*E *7/*F";

static  char  txt[ALL_ROWS][COLS+1];

static  int  offs_x=0,offs_y=0,vis_x,vis_y;
static  int  wind_x,wind_y,wind_w=-1,wind_h;

/* === Functions ==============================================	*/

int prf_open(EMU_COMBO *emulation) {
  /*------------------------------*/
  int  x,y,w,h,handle;

  wind_get(0,WF_WORKXYWH,&x,&y,&w,&h);
  if (wind_w<0) {
    wind_calc(WC_WORK,KIND,x,y,w,h/2,
              &wind_x,&wind_y,&wind_w,&wind_h);
    wind_x=((wind_x+wchr/2)/wchr)*wchr;
    wind_w=((wind_w+wchr/2)/wchr)*wchr;
    wind_h=((wind_h+hchr/2)/hchr)*hchr;
    if (wind_w>COLS*wchr) {
      wind_x+=wind_w-COLS*wchr;
      wind_w=COLS*wchr;
    }
    if (wind_h>(ALL_ROWS)*hchr)
      wind_h=(ALL_ROWS)*hchr;
  }
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  x=((x+wchr/2)/wchr)*wchr;
  w=((w+wchr/2)/wchr)*wchr;
  h=((h+hchr/2)/hchr)*hchr;
  if (w>COLS*wchr) {
    x+=w-COLS*wchr;
    w=COLS*wchr;
  }
  if (h>(ALL_ROWS)*hchr)
    h=(ALL_ROWS)*hchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  if ((handle=wind_create(KIND,x,y,w,h))<0)
    return -3;			/* can't create window */

  redraw(handle,emulation);

  wind_set(handle,WF_INFO,info);
  wind_calc(WC_BORDER,KIND,wind_x,wind_y,wind_w,wind_h,
            &x,&y,&w,&h);
  graf_mouse(M_OFF,0L);
  graf_growbox(0,0,0,0,x,y,w,h);
  wind_open(handle,x,y,w,h);
  graf_mouse(M_ON,0L);
  set_slider(handle);
  return handle;
}

void prf_close(int handle) {
  /*----------------------*/
  int  x,y,w,h;

  wind_get(handle,WF_WORKXYWH,&wind_x,&wind_y,&wind_w,&wind_h);
  wind_get(handle,WF_CURRXYWH,&x,&y,&w,&h);
  graf_shrinkbox(0,0,0,0,x,y,w,h);
  wind_close(handle);
  wind_delete(handle);
}

void prf_redraw(int handle,int rx,int ry,int rw,int rh) {
  /*---------------------------------------------------*/
  int  x,y,w,h,wx,wy,ww,wh,yy,i;
  int  pxy[8];

  wind_set(handle,WF_INFO,info+offs_x);

  graf_mouse(M_OFF,0L);
  if (rw<0)
    wind_get(handle,WF_WORKXYWH,&rx,&ry,&rw,&rh);
  wind_get(handle,WF_WORKXYWH,&wx,&wy,&ww,&wh);
  wind_get(handle,WF_FIRSTXYWH,&x,&y,&w,&h);
  while (w!=0 && h!=0) {
    if (rc_intersect(rx,ry,rw,rh,&x,&y,&w,&h)) {
      pxy[0]=x;
      pxy[1]=y;
      pxy[2]=x+w-1;
      pxy[3]=y+h-1;
      vs_clip(vdi_handle,1,pxy);

      for (yy=0, i=offs_y; yy<wh; yy+=hchr, i++) {
        v_gtext(vdi_handle,wx,wy+yy,txt[i]+offs_x);
        vqt_extent(vdi_handle,txt[i]+offs_x,pxy);
        pxy[0]=wx+pxy[2];
        pxy[1]=wy+yy+pxy[3];
        pxy[2]=wx+ww-1;
        pxy[3]=pxy[1]+hchr-1;
        if (pxy[0]<=pxy[2])
          v_bar(vdi_handle,pxy);
      }
    }
    wind_get(handle,WF_NEXTXYWH,&x,&y,&w,&h);
  }
  graf_mouse(M_ON,0L);
}

void prf_sized(int handle,int x,int y,int w,int h) {
  /*----------------------------------------------*/
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  w=((w+wchr/2)/wchr)*wchr;
  if (w>COLS*wchr)
    w=COLS*wchr;
  h=((h+hchr/2)/hchr)*hchr;
  if (h>(ALL_ROWS)*hchr)
    h=(ALL_ROWS)*hchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  wind_set(handle,WF_CURRXYWH,x,y,w,h);
  set_slider(handle);
  prf_redraw(handle,0,0,-1,0);
}

void prf_fulled(int handle) {
  /*-----------------------*/
  int  cx,cy,cw,ch,x,y,w,h;

  wind_get(handle,WF_CURRXYWH,&cx,&cy,&cw,&ch);
  wind_get(handle,WF_FULLXYWH,&x,&y,&w,&h);
  if (cx==x && cy==y && cw==w && ch==h)
    wind_get(handle,WF_PREVXYWH,&x,&y,&w,&h);
  prf_sized(handle,x,y,w,h);
}

void prf_moved(int handle,int x,int y,int w,int h) {
  /*----------------------------------------------*/
  wind_calc(WC_WORK,KIND,x,y,w,h,&x,&y,&w,&h);
  x=((x+wchr/2)/wchr)*wchr;
  wind_calc(WC_BORDER,KIND,x,y,w,h,&x,&y,&w,&h);
  wind_set(handle,WF_CURRXYWH,x,y,w,h);
}

void prf_vslided(int handle,int pos) {
  /*--------------------------------*/
  offs_y=(int)((long)pos*((ALL_ROWS)-vis_y)/1000);
  if (offs_y<0)
    offs_y=0;
  set_slider(handle);
  prf_redraw(handle,0,0,-1,0);
}

void prf_hslided(int handle,int pos) {
  /*--------------------------------*/
  offs_x=(int)((long)pos*(COLS-vis_x)/1000);
  if (offs_x<0)
    offs_x=0;
  set_slider(handle);
  prf_redraw(handle,0,0,-1,0);
}

void prf_arrowed(int handle,int arrow) {
  /*----------------------------------*/
  int  i;

  i=1;
  switch (arrow) {
  case WA_UPPAGE:
    i=vis_y;
  case WA_UPLINE:
    while (i--) {
      if (offs_y==0)
        break;
      offs_y--;
    }
    break;
  case WA_DNPAGE:
    i=vis_y;
  case WA_DNLINE:
    while (i--) {
      if (offs_y>=(ALL_ROWS)-vis_y)
        break;
      offs_y++;
    }
    break;
  case WA_LFPAGE:
    i=vis_x;
  case WA_LFLINE:
    while (i--) {
      if (offs_x==0)
        break;
      offs_x--;
    }
    break;
  case WA_RTPAGE:
    i=vis_x;
  case WA_RTLINE:
    while (i--) {
      if (offs_x>=COLS-vis_x)
        break;
      offs_x++;
    }
    break;
  default:
    return;
  }

  set_slider(handle);
  prf_redraw(handle,0,0,-1,0);
}

void prf_update(int handle,EMU_COMBO *emulation,int clear) {
  /*------------------------------------------------------*/
  int  i;

  if (clear) {
    for (i=-0x80; i<+0x80; i++)
      emulation[i].prof=0;
    emu_time=0;
    emu_frames=0;
  }

  redraw(handle,emulation);

  if (handle>=0) {
    set_slider(handle);
    prf_redraw(handle,0,0,-1,0);
  }
}

static void set_slider(int handle) {
  /*------------------------------*/
  int  p,s,dummy;

  wind_get(handle,WF_WORKXYWH,&dummy,&dummy,&vis_x,&vis_y);

  vis_y/=hchr;
  if ((ALL_ROWS)<=vis_y) {
    offs_y=0;
    p=0;
    s=1000;
  } else {
    if (offs_y>(ALL_ROWS)-vis_y)
      offs_y=(ALL_ROWS)-vis_y;
    p=(int)(1000L*offs_y/((ALL_ROWS)-vis_y));
    s=(int)(1000L*vis_y/(ALL_ROWS));
  }
  wind_set(handle,WF_VSLIDE,p);
  wind_set(handle,WF_VSLSIZE,s);

  vis_x/=wchr;
  if (COLS<=vis_x) {
    offs_x=0;
    p=0;
    s=1000;
  } else {
    if (offs_x>COLS-vis_x)
      offs_x=COLS-vis_x;
    p=(int)(1000L*offs_x/(COLS-vis_x));
    s=(int)(1000L*vis_x/COLS);
  }
  wind_set(handle,WF_HSLIDE,p);
  wind_set(handle,WF_HSLSIZE,s);
}

#define PRF(x) emulation[(x)].prof

static void redraw(int handle,EMU_COMBO *emulation) {
  /*-----------------------------------------------*/
  int          i,j;
  float        sum;
  long         rd_a,wr_a,rd_b,wr_b,rd_c,wr_c,rd_d,wr_d,rd_e,wr_e,
               rd_h,wr_h,rd_l,wr_l,rd_m,wr_m,push,pop,jp;
  char         *s;
  long         min,sec;
  long         fps;
  static char  title[COLS+1];

  for (sum=0, i=-0x80; i<+0x80; sum+=PRF(i++));

  rd_a=PRF(0x02)+PRF(0x07)+PRF(0x0f)
      +PRF(0x12)+PRF(0x17)+PRF(0x1f)
      +PRF(0x22)+PRF(0x2f)
      +PRF(0x32)+PRF(0x3c)+PRF(0x3d);
  wr_a=PRF(0x07)+PRF(0x0a)+PRF(0x0f)
      +PRF(0x17)+PRF(0x1a)+PRF(0x1f)
      +PRF(0x2a)+PRF(0x2f)
      +PRF(0x3a)+PRF(0x3c)+PRF(0x3d)+PRF(0x3e);
  rd_b=PRF(0x02)+PRF(0x03)+PRF(0x04)+PRF(0x05)+PRF(0x09)
      +PRF(0x0a)+PRF(0x0b);
  wr_b=PRF(0x01)+PRF(0x03)+PRF(0x04)+PRF(0x05)+PRF(0x06)
      +PRF(0x0b);
  rd_c=PRF(0x02)+PRF(0x03)+PRF(0x09)+PRF(0x0a)+PRF(0x0b)
      +PRF(0x0c)+PRF(0x0d);
  wr_c=PRF(0x01)+PRF(0x03)+PRF(0x0b)+PRF(0x0c)+PRF(0x0d)
      +PRF(0x0e);
  rd_d=PRF(0x12)+PRF(0x13)+PRF(0x14)+PRF(0x15)+PRF(0x19)
      +PRF(0x1a)+PRF(0x1b);
  wr_d=PRF(0x11)+PRF(0x13)+PRF(0x14)+PRF(0x15)+PRF(0x16)
      +PRF(0x1b);
  rd_e=PRF(0x12)+PRF(0x13)+PRF(0x19)+PRF(0x1a)+PRF(0x1b)
      +PRF(0x1c)+PRF(0x1d);
  wr_e=PRF(0x11)+PRF(0x13)+PRF(0x1b)+PRF(0x1c)+PRF(0x1d)
      +PRF(0x1e);
  rd_h=PRF(0x09)+PRF(0x19)
      +PRF(0x22)+PRF(0x23)+PRF(0x24)+PRF(0x25)+PRF(0x26)
      +PRF(0x29)+PRF(0x2a)+PRF(0x2b)
      +PRF(0x32)+PRF(0x36)+PRF(0x39)+PRF(0x3a);
  wr_h=PRF(0x09)+PRF(0x19)
      +PRF(0x21)+PRF(0x22)+PRF(0x23)+PRF(0x24)+PRF(0x25)
      +PRF(0x29)+PRF(0x2a)+PRF(0x2b)
      +PRF(0x32)+PRF(0x39)+PRF(0x3a);
  rd_l=PRF(0x09)+PRF(0x19)
      +PRF(0x22)+PRF(0x23)+PRF(0x29)+PRF(0x2a)+PRF(0x2b)
      +PRF(0x2c)+PRF(0x2d)+PRF(0x32)
      +PRF(0x36)+PRF(0x39)+PRF(0x3a);
  wr_l=PRF(0x09)+PRF(0x19)
      +PRF(0x21)+PRF(0x22)+PRF(0x23)+PRF(0x29)+PRF(0x2a)
      +PRF(0x2b)+PRF(0x2c)+PRF(0x2d)+PRF(0x2e)
      +PRF(0x32)+PRF(0x39)+PRF(0x3a);
  rd_m=PRF(0x0a)+PRF(0x1a)+PRF(0x2a)
      +PRF(0x34)+PRF(0x35)+PRF(0x3a);
  wr_m=PRF(0x02)+PRF(0x08)+PRF(0x12)+PRF(0x22)
      +PRF(0x32)+PRF(0x34)+PRF(0x35)+PRF(0x36);
  for (i=0x47; i<=0x7f; i+=8)  rd_a+=PRF(i);
  for (i=0x78; i<=0x7f; i++)   wr_a+=PRF(i);
  for (i=0x40; i<=0x78; i+=8)  rd_b+=PRF(i);
  for (i=0x40; i<=0x47; i++)   wr_b+=PRF(i);
  for (i=0x41; i<=0x79; i+=8)  rd_c+=PRF(i);
  for (i=0x48; i<=0x4f; i++)   wr_c+=PRF(i);
  for (i=0x42; i<=0x7a; i+=8)  rd_d+=PRF(i);
  for (i=0x50; i<=0x57; i++)   wr_d+=PRF(i);
  for (i=0x43; i<=0x7b; i+=8)  rd_e+=PRF(i);
  for (i=0x58; i<=0x5f; i++)   wr_e+=PRF(i);
  for (i=0x44; i<=0x7c; i+=8)  rd_h+=PRF(i);
  for (i=0x60; i<=0x67; i++)   wr_h+=PRF(i);
  for (i=0x45; i<=0x7d; i+=8)  rd_l+=PRF(i);
  for (i=0x68; i<=0x6f; i++)   wr_l+=PRF(i);
  for (i=0x46; i<=0x7e; i+=8)  rd_m+=PRF(i);
  rd_m-=PRF(0x76);
  for (i=0x70; i<=0x77; i++)   wr_m+=PRF(i);
  wr_m-=PRF(0x76);
  for (i=0x80; i<=0xbf; i++)   rd_a+=PRF(i-0x100);
  for (i=0x87; i<=0xbf; i+=8)  rd_a+=PRF(i-0x100);
  for (i=0x80; i<=0xb8; i+=8)  rd_b+=PRF(i-0x100);
  for (i=0x81; i<=0xb9; i+=8)  rd_c+=PRF(i-0x100);
  for (i=0x82; i<=0xba; i+=8)  rd_d+=PRF(i-0x100);
  for (i=0x83; i<=0xbb; i+=8)  rd_e+=PRF(i-0x100);
  for (i=0x84; i<=0xbc; i+=8)  rd_h+=PRF(i-0x100);
  for (i=0x85; i<=0xbd; i+=8)  rd_l+=PRF(i-0x100);
  for (i=0x86; i<=0xbe; i+=8)  rd_m+=PRF(i-0x100);
  for (i=0x80; i<=0xb7; i++)   wr_a+=PRF(i-0x100);
  rd_a+=PRF(0xc6-0x100)+PRF(0xce-0x100)
       +PRF(0xd6-0x100)+PRF(0xde-0x100)
       +PRF(0xe2-0x100)+PRF(0xe6-0x100)+PRF(0xea-0x100)
       +PRF(0xee-0x100)
       +PRF(0xf5-0x100)+PRF(0xf6-0x100)+PRF(0xfe-0x100);
  wr_a+=PRF(0xc6-0x100)+PRF(0xce-0x100)
       +PRF(0xd6-0x100)+PRF(0xde-0x100)
       +PRF(0xe6-0x100)+PRF(0xee-0x100)
       +PRF(0xf0-0x100)+PRF(0xf1-0x100)+PRF(0xf2-0x100)
       +PRF(0xf6-0x100)+PRF(0xfa-0x100);
  rd_b+=PRF(0xc5-0x100);
  wr_b+=PRF(0xc1-0x100);
  rd_c+=PRF(0xc5-0x100)+PRF(0xe2-0x100)+PRF(0xf2-0x100);
  wr_c+=PRF(0xc1-0x100);
  rd_d+=PRF(0xd5-0x100);
  wr_d+=PRF(0xd1-0x100);
  rd_e+=PRF(0xd5-0x100);
  wr_e+=PRF(0xd1-0x100);
  rd_h+=PRF(0xe5-0x100)+PRF(0xe9-0x100)+PRF(0xf9-0x100);
  wr_h+=PRF(0xe1-0x100)+PRF(0xf8-0x100);
  rd_l+=PRF(0xe5-0x100)+PRF(0xe9-0x100)+PRF(0xf9-0x100);
  wr_l+=PRF(0xe1-0x100)+PRF(0xf8-0x100);
  rd_m+=PRF(0xf0-0x100)+PRF(0xf2-0x100)+PRF(0xfa-0x100);
  wr_m+=PRF(0xe0-0x100)+PRF(0xe2-0x100)+PRF(0xea-0x100);
  push=PRF(0xc4-0x100)+PRF(0xc5-0x100)+PRF(0xc7-0x100)
      +PRF(0xcc-0x100)+PRF(0xcd-0x100)+PRF(0xcf-0x100)
      +PRF(0xd4-0x100)+PRF(0xd5-0x100)+PRF(0xd7-0x100)
      +PRF(0xdc-0x100)+PRF(0xdf-0x100)
      +PRF(0xe5-0x100)+PRF(0xe7-0x100)+PRF(0xef-0x100)
      +PRF(0xf5-0x100)+PRF(0xf7-0x100)+PRF(0xff-0x100);
  pop=PRF(0xc0-0x100)+PRF(0xc1-0x100)
     +PRF(0xc8-0x100)+PRF(0xc9-0x100)
     +PRF(0xd0-0x100)+PRF(0xd1-0x100)
     +PRF(0xd8-0x100)+PRF(0xd9-0x100)
     +PRF(0xe1-0x100)+PRF(0xf1-0x100);
  jp=PRF(0xc2-0x100)+PRF(0xc3-0x100)+PRF(0xc4-0x100)
    +PRF(0xca-0x100)+PRF(0xcc-0x100)+PRF(0xcd-0x100)
    +PRF(0xd2-0x100)+PRF(0xd4-0x100)
    +PRF(0xda-0x100)+PRF(0xdc-0x100)+PRF(0xe9-0x100);

  if (handle>=0) {
    rsrc_gaddr(R_STRING,SPRFTITL,&s);
    min=(emu_time+CLK_TCK/2)/CLK_TCK;
    if (min==0)
      fps=0;
    else
      fps=(emu_frames+min/2)/min;
    sec=min%60;
    min/=60;
    sprintf(title,s,sum,min,sec,fps);
    wind_set(handle,WF_NAME,title);
  }

  sum/=100;
  if (sum==0)
    sum=1;
  for (i=0; i<0x8; i++) {
    sprintf(txt[0x10/DIV*i],"%X*"
      FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT,
      i,
      (float)PRF(0x10*i+0x0)/sum,
      (float)PRF(0x10*i+0x1)/sum,
      (float)PRF(0x10*i+0x2)/sum,
      (float)PRF(0x10*i+0x3)/sum,
      (float)PRF(0x10*i+0x4)/sum,
      (float)PRF(0x10*i+0x5)/sum,
      (float)PRF(0x10*i+0x6)/sum,
      (float)PRF(0x10*i+0x7)/sum);
    sprintf(txt[0x10/DIV*i+1],"  "
      FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT,
      (float)PRF(0x10*i+0x8)/sum,
      (float)PRF(0x10*i+0x9)/sum,
      (float)PRF(0x10*i+0xa)/sum,
      (float)PRF(0x10*i+0xb)/sum,
      (float)PRF(0x10*i+0xc)/sum,
      (float)PRF(0x10*i+0xd)/sum,
      (float)PRF(0x10*i+0xe)/sum,
      (float)PRF(0x10*i+0xf)/sum);
  }
  for (; i<0x10; i++) {
    sprintf(txt[0x10/DIV*i],"%X*"
      FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT,
      i,
      (float)PRF(0x10*i+0x0-0x100)/sum,
      (float)PRF(0x10*i+0x1-0x100)/sum,
      (float)PRF(0x10*i+0x2-0x100)/sum,
      (float)PRF(0x10*i+0x3-0x100)/sum,
      (float)PRF(0x10*i+0x4-0x100)/sum,
      (float)PRF(0x10*i+0x5-0x100)/sum,
      (float)PRF(0x10*i+0x6-0x100)/sum,
      (float)PRF(0x10*i+0x7-0x100)/sum);
    sprintf(txt[0x10/DIV*i+1],"  "
      FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT FORMAT,
      (float)PRF(0x10*i+0x8-0x100)/sum,
      (float)PRF(0x10*i+0x9-0x100)/sum,
      (float)PRF(0x10*i+0xa-0x100)/sum,
      (float)PRF(0x10*i+0xb-0x100)/sum,
      (float)PRF(0x10*i+0xc-0x100)/sum,
      (float)PRF(0x10*i+0xd-0x100)/sum,
      (float)PRF(0x10*i+0xe-0x100)/sum,
      (float)PRF(0x10*i+0xf-0x100)/sum);
  }
  for (i=0; i<0x80; i+=DIV) {
    s=txt[i/DIV]+2;
    for (j=0; j<DIV; j++) {
      if (PRF(i+j)==0)
        strncpy(s,F_EMPTY,F_SIZE);
      s+=F_SIZE;
    }
  }
  for (; i<0x100; i+=DIV) {
    s=txt[i/DIV]+2;
    for (j=0; j<DIV; j++) {
      if (PRF(i+j-0x100)==0)
        strncpy(s,F_EMPTY,F_SIZE);
      s+=F_SIZE;
    }
  }
  sprintf(txt[ROWS+ 0],"A(r) " FORMAT "%%=%10ld",
    (float)rd_a/sum,rd_a);
  sprintf(txt[ROWS+ 1],"A(w) " FORMAT "%%=%10ld",
    (float)wr_a/sum,wr_a);
  sprintf(txt[ROWS+ 2],"B(r) " FORMAT "%%=%10ld",
    (float)rd_b/sum,rd_b);
  sprintf(txt[ROWS+ 3],"B(w) " FORMAT "%%=%10ld",
    (float)wr_b/sum,wr_b);
  sprintf(txt[ROWS+ 4],"C(r) " FORMAT "%%=%10ld",
    (float)rd_c/sum,rd_c);
  sprintf(txt[ROWS+ 5],"C(w) " FORMAT "%%=%10ld",
    (float)wr_c/sum,wr_c);
  sprintf(txt[ROWS+ 6],"D(r) " FORMAT "%%=%10ld",
    (float)rd_d/sum,rd_d);
  sprintf(txt[ROWS+ 7],"D(w) " FORMAT "%%=%10ld",
    (float)wr_d/sum,wr_d);
  sprintf(txt[ROWS+ 8],"E(r) " FORMAT "%%=%10ld",
    (float)rd_e/sum,rd_e);
  sprintf(txt[ROWS+ 9],"E(w) " FORMAT "%%=%10ld",
    (float)wr_e/sum,wr_e);
  sprintf(txt[ROWS+10],"H(r) " FORMAT "%%=%10ld",
    (float)rd_h/sum,rd_h);
  sprintf(txt[ROWS+11],"H(w) " FORMAT "%%=%10ld",
    (float)wr_h/sum,wr_h);
  sprintf(txt[ROWS+12],"L(r) " FORMAT "%%=%10ld",
    (float)rd_l/sum,rd_l);
  sprintf(txt[ROWS+13],"L(w) " FORMAT "%%=%10ld",
    (float)wr_l/sum,wr_l);
  sprintf(txt[ROWS+14],"M(r) " FORMAT "%%=%10ld",
    (float)rd_m/sum,rd_m);
  sprintf(txt[ROWS+15],"M(w) " FORMAT "%%=%10ld",
    (float)wr_m/sum,wr_m);
  sprintf(txt[ROWS+16],"PUSH " FORMAT "%%=%10ld",
    (float)push/sum,push);
  sprintf(txt[ROWS+17],"POP  " FORMAT "%%=%10ld",
    (float)pop/sum,pop);
  sprintf(txt[ROWS+18],"JP   " FORMAT "%%=%10ld",
    (float)jp/sum,jp);
#if ACC_ROWS!=19 /* count of longs */
#error "ACC_ROWS isn't appropriate!"
#endif
}

/* === End ====================================================	*/
